/****************************************************************************
**
** Copyright (C) 2000-2008 TROLLTECH ASA. All rights reserved.
**
** This file is part of the Opensource Edition of the Qtopia Toolkit.
**
** This software is licensed under the terms of the GNU General Public
** License (GPL) version 2.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "dtmfstream.h"
#include <math.h>

// This class should appear in the media stream just after
// AudioInputStream.  As audio data is streamed from the
// microphone, the tones are automatically mixed in.

DtmfStream::DtmfStream( QObject *parent )
    : MediaStream( parent )
{
    generatingTones = false;
    open( WriteOnly );
}

DtmfStream::~DtmfStream()
{
}

void DtmfStream::dtmf( const QString& tones )
{
    pendingTones += tones;
    if ( !generatingTones )
        nextTone();
}

qint64 DtmfStream::writeData( const char *data, qint64 len )
{
    // If we are not generating DTMF tones at the moment,
    // then just pass the data onto the next object in the stream.
    if ( !generatingTones )
        return writeNext( data, len );

    // Generate DTMF samples into the buffer.  We assume that the
    // buffer was given to us by AudioInputStream and that it will
    // not care if we destroy its data.
    short *buffer = (short *)data;
    qint64 left = len / 2;
    while ( ( samplesToGenerate > 0 || gapSamples > 0 ) && left > 0 ) {
        if ( samplesToGenerate > 0 ) {
            *buffer++ = sine[ ( offset * lower ) % 8000 ] +
                        sine[ ( offset * higher ) % 8000 ];
            --samplesToGenerate;
        } else {
            ++buffer;
            --gapSamples;
        }
        ++offset;
        --left;
    }

    // If we have finished generating this tone, then move on to the next one.
    if ( samplesToGenerate == 0 && gapSamples == 0 ) {
        nextTone();
    }

    // Write the modified packet to the output stream.
    return writeNext( data, len );
}

// Get the frequency values for the next pending tone.  Returns
// false if there are no more tones to be generated.
bool DtmfStream::getToneFrequencies( int& lower, int& higher )
{
    QChar ch;
    while ( pendingTones.length() > 0 ) {
        ch = pendingTones[0];
        pendingTones = pendingTones.mid(1);
        switch ( ch.unicode() ) {

            case '0':           lower = 941; higher = 1336; return true;
            case '1':           lower = 697; higher = 1209; return true;
            case '2':           lower = 697; higher = 1336; return true;
            case '3':           lower = 697; higher = 1477; return true;
            case '4':           lower = 770; higher = 1209; return true;
            case '5':           lower = 770; higher = 1336; return true;
            case '6':           lower = 770; higher = 1477; return true;
            case '7':           lower = 852; higher = 1209; return true;
            case '8':           lower = 852; higher = 1336; return true;
            case '9':           lower = 852; higher = 1477; return true;
            case 'a': case 'A': lower = 697; higher = 1633; return true;
            case 'b': case 'B': lower = 770; higher = 1633; return true;
            case 'c': case 'C': lower = 852; higher = 1633; return true;
            case 'd': case 'D': lower = 941; higher = 1633; return true;
            case '*': case 'e': case 'E':
                                lower = 941; higher = 1209; return true;
            case '#': case 'f': case 'F':
                                lower = 941; higher = 1477; return true;
            case 'x': case 'X': lower =  20; higher =   20; return true;
            case 'y': case 'Y': lower = 100; higher =  100; return true;
            case 'z': case 'Z': lower = 500; higher =  500; return true;
            default:            break;

        }
    }
    return false;
}

void DtmfStream::nextTone()
{
    // Get the frequencies for the next tone.
    if ( !getToneFrequencies( lower, higher ) ) {
        generatingTones = false;
        return;
    }

    // Generate 200 ms worth of tone, followed by 100 ms worth of gap.
    offset = 0;
    samplesToGenerate = 1600;
    gapSamples = 800;
    generatingTones = true;
}

// Pre-computed table of amplitude values for a 1 Hz sine wave.
// See below for the code that generates this table.
short const DtmfStream::sine[8000] = {
 0, 1, 3, 4, 6, 8, 9, 11,
 12, 14, 16, 17, 19, 20, 22, 24,
 25, 27, 28, 30, 32, 33, 35, 36,
 38, 40, 41, 43, 45, 46, 48, 49,
 51, 53, 54, 56, 57, 59, 61, 62,
 64, 65, 67, 69, 70, 72, 73, 75,
 77, 78, 80, 82, 83, 85, 86, 88,
 90, 91, 93, 94, 96, 98, 99, 101,
 102, 104, 106, 107, 109, 110, 112, 114,
 115, 117, 118, 120, 122, 123, 125, 126,
 128, 130, 131, 133, 135, 136, 138, 139,
 141, 143, 144, 146, 147, 149, 151, 152,
 154, 155, 157, 159, 160, 162, 163, 165,
 167, 168, 170, 171, 173, 175, 176, 178,
 179, 181, 183, 184, 186, 187, 189, 191,
 192, 194, 195, 197, 199, 200, 202, 203,
 205, 207, 208, 210, 211, 213, 215, 216,
 218, 219, 221, 223, 224, 226, 227, 229,
 231, 232, 234, 235, 237, 239, 240, 242,
 243, 245, 247, 248, 250, 251, 253, 255,
 256, 258, 259, 261, 263, 264, 266, 267,
 269, 271, 272, 274, 275, 277, 279, 280,
 282, 283, 285, 286, 288, 290, 291, 293,
 294, 296, 298, 299, 301, 302, 304, 306,
 307, 309, 310, 312, 314, 315, 317, 318,
 320, 321, 323, 325, 326, 328, 329, 331,
 333, 334, 336, 337, 339, 341, 342, 344,
 345, 347, 348, 350, 352, 353, 355, 356,
 358, 360, 361, 363, 364, 366, 367, 369,
 371, 372, 374, 375, 377, 379, 380, 382,
 383, 385, 386, 388, 390, 391, 393, 394,
 396, 397, 399, 401, 402, 404, 405, 407,
 409, 410, 412, 413, 415, 416, 418, 420,
 421, 423, 424, 426, 427, 429, 431, 432,
 434, 435, 437, 438, 440, 442, 443, 445,
 446, 448, 449, 451, 453, 454, 456, 457,
 459, 460, 462, 464, 465, 467, 468, 470,
 471, 473, 474, 476, 478, 479, 481, 482,
 484, 485, 487, 489, 490, 492, 493, 495,
 496, 498, 499, 501, 503, 504, 506, 507,
 509, 510, 512, 513, 515, 517, 518, 520,
 521, 523, 524, 526, 527, 529, 531, 532,
 534, 535, 537, 538, 540, 541, 543, 545,
 546, 548, 549, 551, 552, 554, 555, 557,
 559, 560, 562, 563, 565, 566, 568, 569,
 571, 572, 574, 576, 577, 579, 580, 582,
 583, 585, 586, 588, 589, 591, 592, 594,
 596, 597, 599, 600, 602, 603, 605, 606,
 608, 609, 611, 612, 614, 616, 617, 619,
 620, 622, 623, 625, 626, 628, 629, 631,
 632, 634, 635, 637, 638, 640, 642, 643,
 645, 646, 648, 649, 651, 652, 654, 655,
 657, 658, 660, 661, 663, 664, 666, 667,
 669, 670, 672, 674, 675, 677, 678, 680,
 681, 683, 684, 686, 687, 689, 690, 692,
 693, 695, 696, 698, 699, 701, 702, 704,
 705, 707, 708, 710, 711, 713, 714, 716,
 717, 719, 720, 722, 723, 725, 726, 728,
 729, 731, 732, 734, 735, 737, 738, 740,
 741, 743, 744, 746, 747, 749, 750, 752,
 753, 755, 756, 758, 759, 761, 762, 764,
 765, 767, 768, 770, 771, 773, 774, 776,
 777, 779, 780, 782, 783, 785, 786, 788,
 789, 791, 792, 794, 795, 797, 798, 800,
 801, 803, 804, 805, 807, 808, 810, 811,
 813, 814, 816, 817, 819, 820, 822, 823,
 825, 826, 828, 829, 831, 832, 833, 835,
 836, 838, 839, 841, 842, 844, 845, 847,
 848, 850, 851, 853, 854, 855, 857, 858,
 860, 861, 863, 864, 866, 867, 869, 870,
 871, 873, 874, 876, 877, 879, 880, 882,
 883, 885, 886, 887, 889, 890, 892, 893,
 895, 896, 898, 899, 900, 902, 903, 905,
 906, 908, 909, 911, 912, 913, 915, 916,
 918, 919, 921, 922, 924, 925, 926, 928,
 929, 931, 932, 934, 935, 936, 938, 939,
 941, 942, 944, 945, 946, 948, 949, 951,
 952, 954, 955, 956, 958, 959, 961, 962,
 964, 965, 966, 968, 969, 971, 972, 973,
 975, 976, 978, 979, 980, 982, 983, 985,
 986, 988, 989, 990, 992, 993, 995, 996,
 997, 999, 1000, 1002, 1003, 1004, 1006, 1007,
 1009, 1010, 1011, 1013, 1014, 1016, 1017, 1018,
 1020, 1021, 1023, 1024, 1025, 1027, 1028, 1030,
 1031, 1032, 1034, 1035, 1036, 1038, 1039, 1041,
 1042, 1043, 1045, 1046, 1048, 1049, 1050, 1052,
 1053, 1054, 1056, 1057, 1059, 1060, 1061, 1063,
 1064, 1065, 1067, 1068, 1070, 1071, 1072, 1074,
 1075, 1076, 1078, 1079, 1081, 1082, 1083, 1085,
 1086, 1087, 1089, 1090, 1091, 1093, 1094, 1096,
 1097, 1098, 1100, 1101, 1102, 1104, 1105, 1106,
 1108, 1109, 1110, 1112, 1113, 1114, 1116, 1117,
 1119, 1120, 1121, 1123, 1124, 1125, 1127, 1128,
 1129, 1131, 1132, 1133, 1135, 1136, 1137, 1139,
 1140, 1141, 1143, 1144, 1145, 1147, 1148, 1149,
 1151, 1152, 1153, 1155, 1156, 1157, 1159, 1160,
 1161, 1163, 1164, 1165, 1167, 1168, 1169, 1171,
 1172, 1173, 1174, 1176, 1177, 1178, 1180, 1181,
 1182, 1184, 1185, 1186, 1188, 1189, 1190, 1192,
 1193, 1194, 1195, 1197, 1198, 1199, 1201, 1202,
 1203, 1205, 1206, 1207, 1208, 1210, 1211, 1212,
 1214, 1215, 1216, 1218, 1219, 1220, 1221, 1223,
 1224, 1225, 1227, 1228, 1229, 1230, 1232, 1233,
 1234, 1236, 1237, 1238, 1239, 1241, 1242, 1243,
 1245, 1246, 1247, 1248, 1250, 1251, 1252, 1253,
 1255, 1256, 1257, 1259, 1260, 1261, 1262, 1264,
 1265, 1266, 1267, 1269, 1270, 1271, 1272, 1274,
 1275, 1276, 1277, 1279, 1280, 1281, 1283, 1284,
 1285, 1286, 1288, 1289, 1290, 1291, 1293, 1294,
 1295, 1296, 1297, 1299, 1300, 1301, 1302, 1304,
 1305, 1306, 1307, 1309, 1310, 1311, 1312, 1314,
 1315, 1316, 1317, 1319, 1320, 1321, 1322, 1323,
 1325, 1326, 1327, 1328, 1330, 1331, 1332, 1333,
 1334, 1336, 1337, 1338, 1339, 1341, 1342, 1343,
 1344, 1345, 1347, 1348, 1349, 1350, 1351, 1353,
 1354, 1355, 1356, 1357, 1359, 1360, 1361, 1362,
 1363, 1365, 1366, 1367, 1368, 1369, 1371, 1372,
 1373, 1374, 1375, 1377, 1378, 1379, 1380, 1381,
 1383, 1384, 1385, 1386, 1387, 1389, 1390, 1391,
 1392, 1393, 1394, 1396, 1397, 1398, 1399, 1400,
 1401, 1403, 1404, 1405, 1406, 1407, 1408, 1410,
 1411, 1412, 1413, 1414, 1415, 1417, 1418, 1419,
 1420, 1421, 1422, 1424, 1425, 1426, 1427, 1428,
 1429, 1430, 1432, 1433, 1434, 1435, 1436, 1437,
 1439, 1440, 1441, 1442, 1443, 1444, 1445, 1447,
 1448, 1449, 1450, 1451, 1452, 1453, 1454, 1456,
 1457, 1458, 1459, 1460, 1461, 1462, 1463, 1465,
 1466, 1467, 1468, 1469, 1470, 1471, 1472, 1474,
 1475, 1476, 1477, 1478, 1479, 1480, 1481, 1482,
 1484, 1485, 1486, 1487, 1488, 1489, 1490, 1491,
 1492, 1494, 1495, 1496, 1497, 1498, 1499, 1500,
 1501, 1502, 1503, 1504, 1506, 1507, 1508, 1509,
 1510, 1511, 1512, 1513, 1514, 1515, 1516, 1518,
 1519, 1520, 1521, 1522, 1523, 1524, 1525, 1526,
 1527, 1528, 1529, 1530, 1531, 1533, 1534, 1535,
 1536, 1537, 1538, 1539, 1540, 1541, 1542, 1543,
 1544, 1545, 1546, 1547, 1548, 1549, 1551, 1552,
 1553, 1554, 1555, 1556, 1557, 1558, 1559, 1560,
 1561, 1562, 1563, 1564, 1565, 1566, 1567, 1568,
 1569, 1570, 1571, 1572, 1573, 1574, 1575, 1576,
 1578, 1579, 1580, 1581, 1582, 1583, 1584, 1585,
 1586, 1587, 1588, 1589, 1590, 1591, 1592, 1593,
 1594, 1595, 1596, 1597, 1598, 1599, 1600, 1601,
 1602, 1603, 1604, 1605, 1606, 1607, 1608, 1609,
 1610, 1611, 1612, 1613, 1614, 1615, 1616, 1617,
 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625,
 1626, 1627, 1628, 1629, 1629, 1630, 1631, 1632,
 1633, 1634, 1635, 1636, 1637, 1638, 1639, 1640,
 1641, 1642, 1643, 1644, 1645, 1646, 1647, 1648,
 1649, 1650, 1651, 1652, 1653, 1654, 1654, 1655,
 1656, 1657, 1658, 1659, 1660, 1661, 1662, 1663,
 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1670,
 1671, 1672, 1673, 1674, 1675, 1676, 1677, 1678,
 1679, 1680, 1681, 1682, 1682, 1683, 1684, 1685,
 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1692,
 1693, 1694, 1695, 1696, 1697, 1698, 1699, 1700,
 1701, 1701, 1702, 1703, 1704, 1705, 1706, 1707,
 1708, 1709, 1709, 1710, 1711, 1712, 1713, 1714,
 1715, 1716, 1717, 1717, 1718, 1719, 1720, 1721,
 1722, 1723, 1723, 1724, 1725, 1726, 1727, 1728,
 1729, 1730, 1730, 1731, 1732, 1733, 1734, 1735,
 1736, 1736, 1737, 1738, 1739, 1740, 1741, 1741,
 1742, 1743, 1744, 1745, 1746, 1747, 1747, 1748,
 1749, 1750, 1751, 1752, 1752, 1753, 1754, 1755,
 1756, 1757, 1757, 1758, 1759, 1760, 1761, 1761,
 1762, 1763, 1764, 1765, 1766, 1766, 1767, 1768,
 1769, 1770, 1770, 1771, 1772, 1773, 1774, 1774,
 1775, 1776, 1777, 1778, 1778, 1779, 1780, 1781,
 1782, 1782, 1783, 1784, 1785, 1786, 1786, 1787,
 1788, 1789, 1790, 1790, 1791, 1792, 1793, 1793,
 1794, 1795, 1796, 1796, 1797, 1798, 1799, 1800,
 1800, 1801, 1802, 1803, 1803, 1804, 1805, 1806,
 1806, 1807, 1808, 1809, 1809, 1810, 1811, 1812,
 1812, 1813, 1814, 1815, 1815, 1816, 1817, 1818,
 1818, 1819, 1820, 1821, 1821, 1822, 1823, 1824,
 1824, 1825, 1826, 1826, 1827, 1828, 1829, 1829,
 1830, 1831, 1832, 1832, 1833, 1834, 1834, 1835,
 1836, 1837, 1837, 1838, 1839, 1839, 1840, 1841,
 1841, 1842, 1843, 1844, 1844, 1845, 1846, 1846,
 1847, 1848, 1848, 1849, 1850, 1851, 1851, 1852,
 1853, 1853, 1854, 1855, 1855, 1856, 1857, 1857,
 1858, 1859, 1859, 1860, 1861, 1861, 1862, 1863,
 1863, 1864, 1865, 1865, 1866, 1867, 1867, 1868,
 1869, 1869, 1870, 1871, 1871, 1872, 1873, 1873,
 1874, 1875, 1875, 1876, 1876, 1877, 1878, 1878,
 1879, 1880, 1880, 1881, 1882, 1882, 1883, 1884,
 1884, 1885, 1885, 1886, 1887, 1887, 1888, 1889,
 1889, 1890, 1890, 1891, 1892, 1892, 1893, 1893,
 1894, 1895, 1895, 1896, 1896, 1897, 1898, 1898,
 1899, 1900, 1900, 1901, 1901, 1902, 1902, 1903,
 1904, 1904, 1905, 1905, 1906, 1907, 1907, 1908,
 1908, 1909, 1910, 1910, 1911, 1911, 1912, 1912,
 1913, 1914, 1914, 1915, 1915, 1916, 1916, 1917,
 1918, 1918, 1919, 1919, 1920, 1920, 1921, 1921,
 1922, 1923, 1923, 1924, 1924, 1925, 1925, 1926,
 1926, 1927, 1928, 1928, 1929, 1929, 1930, 1930,
 1931, 1931, 1932, 1932, 1933, 1933, 1934, 1934,
 1935, 1936, 1936, 1937, 1937, 1938, 1938, 1939,
 1939, 1940, 1940, 1941, 1941, 1942, 1942, 1943,
 1943, 1944, 1944, 1945, 1945, 1946, 1946, 1947,
 1947, 1948, 1948, 1949, 1949, 1950, 1950, 1951,
 1951, 1952, 1952, 1953, 1953, 1954, 1954, 1955,
 1955, 1956, 1956, 1956, 1957, 1957, 1958, 1958,
 1959, 1959, 1960, 1960, 1961, 1961, 1962, 1962,
 1963, 1963, 1963, 1964, 1964, 1965, 1965, 1966,
 1966, 1967, 1967, 1968, 1968, 1968, 1969, 1969,
 1970, 1970, 1971, 1971, 1971, 1972, 1972, 1973,
 1973, 1974, 1974, 1974, 1975, 1975, 1976, 1976,
 1977, 1977, 1977, 1978, 1978, 1979, 1979, 1980,
 1980, 1980, 1981, 1981, 1982, 1982, 1982, 1983,
 1983, 1984, 1984, 1984, 1985, 1985, 1986, 1986,
 1986, 1987, 1987, 1987, 1988, 1988, 1989, 1989,
 1989, 1990, 1990, 1991, 1991, 1991, 1992, 1992,
 1992, 1993, 1993, 1994, 1994, 1994, 1995, 1995,
 1995, 1996, 1996, 1996, 1997, 1997, 1997, 1998,
 1998, 1999, 1999, 1999, 2000, 2000, 2000, 2001,
 2001, 2001, 2002, 2002, 2002, 2003, 2003, 2003,
 2004, 2004, 2004, 2005, 2005, 2005, 2006, 2006,
 2006, 2007, 2007, 2007, 2008, 2008, 2008, 2008,
 2009, 2009, 2009, 2010, 2010, 2010, 2011, 2011,
 2011, 2012, 2012, 2012, 2012, 2013, 2013, 2013,
 2014, 2014, 2014, 2014, 2015, 2015, 2015, 2016,
 2016, 2016, 2016, 2017, 2017, 2017, 2018, 2018,
 2018, 2018, 2019, 2019, 2019, 2019, 2020, 2020,
 2020, 2020, 2021, 2021, 2021, 2022, 2022, 2022,
 2022, 2023, 2023, 2023, 2023, 2024, 2024, 2024,
 2024, 2024, 2025, 2025, 2025, 2025, 2026, 2026,
 2026, 2026, 2027, 2027, 2027, 2027, 2028, 2028,
 2028, 2028, 2028, 2029, 2029, 2029, 2029, 2029,
 2030, 2030, 2030, 2030, 2031, 2031, 2031, 2031,
 2031, 2032, 2032, 2032, 2032, 2032, 2033, 2033,
 2033, 2033, 2033, 2033, 2034, 2034, 2034, 2034,
 2034, 2035, 2035, 2035, 2035, 2035, 2035, 2036,
 2036, 2036, 2036, 2036, 2037, 2037, 2037, 2037,
 2037, 2037, 2037, 2038, 2038, 2038, 2038, 2038,
 2038, 2039, 2039, 2039, 2039, 2039, 2039, 2039,
 2040, 2040, 2040, 2040, 2040, 2040, 2040, 2041,
 2041, 2041, 2041, 2041, 2041, 2041, 2041, 2042,
 2042, 2042, 2042, 2042, 2042, 2042, 2042, 2042,
 2043, 2043, 2043, 2043, 2043, 2043, 2043, 2043,
 2043, 2044, 2044, 2044, 2044, 2044, 2044, 2044,
 2044, 2044, 2044, 2044, 2045, 2045, 2045, 2045,
 2045, 2045, 2045, 2045, 2045, 2045, 2045, 2045,
 2046, 2046, 2046, 2046, 2046, 2046, 2046, 2046,
 2046, 2046, 2046, 2046, 2046, 2046, 2046, 2046,
 2046, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2048, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2047, 2047, 2047, 2047, 2047, 2047, 2047, 2047,
 2046, 2046, 2046, 2046, 2046, 2046, 2046, 2046,
 2046, 2046, 2046, 2046, 2046, 2046, 2046, 2046,
 2046, 2045, 2045, 2045, 2045, 2045, 2045, 2045,
 2045, 2045, 2045, 2045, 2045, 2044, 2044, 2044,
 2044, 2044, 2044, 2044, 2044, 2044, 2044, 2044,
 2043, 2043, 2043, 2043, 2043, 2043, 2043, 2043,
 2043, 2042, 2042, 2042, 2042, 2042, 2042, 2042,
 2042, 2042, 2041, 2041, 2041, 2041, 2041, 2041,
 2041, 2041, 2040, 2040, 2040, 2040, 2040, 2040,
 2040, 2039, 2039, 2039, 2039, 2039, 2039, 2039,
 2038, 2038, 2038, 2038, 2038, 2038, 2037, 2037,
 2037, 2037, 2037, 2037, 2037, 2036, 2036, 2036,
 2036, 2036, 2035, 2035, 2035, 2035, 2035, 2035,
 2034, 2034, 2034, 2034, 2034, 2033, 2033, 2033,
 2033, 2033, 2033, 2032, 2032, 2032, 2032, 2032,
 2031, 2031, 2031, 2031, 2031, 2030, 2030, 2030,
 2030, 2029, 2029, 2029, 2029, 2029, 2028, 2028,
 2028, 2028, 2028, 2027, 2027, 2027, 2027, 2026,
 2026, 2026, 2026, 2025, 2025, 2025, 2025, 2024,
 2024, 2024, 2024, 2024, 2023, 2023, 2023, 2023,
 2022, 2022, 2022, 2022, 2021, 2021, 2021, 2020,
 2020, 2020, 2020, 2019, 2019, 2019, 2019, 2018,
 2018, 2018, 2018, 2017, 2017, 2017, 2016, 2016,
 2016, 2016, 2015, 2015, 2015, 2014, 2014, 2014,
 2014, 2013, 2013, 2013, 2012, 2012, 2012, 2012,
 2011, 2011, 2011, 2010, 2010, 2010, 2009, 2009,
 2009, 2008, 2008, 2008, 2008, 2007, 2007, 2007,
 2006, 2006, 2006, 2005, 2005, 2005, 2004, 2004,
 2004, 2003, 2003, 2003, 2002, 2002, 2002, 2001,
 2001, 2001, 2000, 2000, 2000, 1999, 1999, 1999,
 1998, 1998, 1997, 1997, 1997, 1996, 1996, 1996,
 1995, 1995, 1995, 1994, 1994, 1994, 1993, 1993,
 1992, 1992, 1992, 1991, 1991, 1991, 1990, 1990,
 1989, 1989, 1989, 1988, 1988, 1987, 1987, 1987,
 1986, 1986, 1986, 1985, 1985, 1984, 1984, 1984,
 1983, 1983, 1982, 1982, 1982, 1981, 1981, 1980,
 1980, 1980, 1979, 1979, 1978, 1978, 1977, 1977,
 1977, 1976, 1976, 1975, 1975, 1974, 1974, 1974,
 1973, 1973, 1972, 1972, 1971, 1971, 1971, 1970,
 1970, 1969, 1969, 1968, 1968, 1968, 1967, 1967,
 1966, 1966, 1965, 1965, 1964, 1964, 1963, 1963,
 1963, 1962, 1962, 1961, 1961, 1960, 1960, 1959,
 1959, 1958, 1958, 1957, 1957, 1956, 1956, 1956,
 1955, 1955, 1954, 1954, 1953, 1953, 1952, 1952,
 1951, 1951, 1950, 1950, 1949, 1949, 1948, 1948,
 1947, 1947, 1946, 1946, 1945, 1945, 1944, 1944,
 1943, 1943, 1942, 1942, 1941, 1941, 1940, 1940,
 1939, 1939, 1938, 1938, 1937, 1937, 1936, 1936,
 1935, 1934, 1934, 1933, 1933, 1932, 1932, 1931,
 1931, 1930, 1930, 1929, 1929, 1928, 1928, 1927,
 1926, 1926, 1925, 1925, 1924, 1924, 1923, 1923,
 1922, 1921, 1921, 1920, 1920, 1919, 1919, 1918,
 1918, 1917, 1916, 1916, 1915, 1915, 1914, 1914,
 1913, 1912, 1912, 1911, 1911, 1910, 1910, 1909,
 1908, 1908, 1907, 1907, 1906, 1905, 1905, 1904,
 1904, 1903, 1902, 1902, 1901, 1901, 1900, 1900,
 1899, 1898, 1898, 1897, 1896, 1896, 1895, 1895,
 1894, 1893, 1893, 1892, 1892, 1891, 1890, 1890,
 1889, 1889, 1888, 1887, 1887, 1886, 1885, 1885,
 1884, 1884, 1883, 1882, 1882, 1881, 1880, 1880,
 1879, 1878, 1878, 1877, 1876, 1876, 1875, 1875,
 1874, 1873, 1873, 1872, 1871, 1871, 1870, 1869,
 1869, 1868, 1867, 1867, 1866, 1865, 1865, 1864,
 1863, 1863, 1862, 1861, 1861, 1860, 1859, 1859,
 1858, 1857, 1857, 1856, 1855, 1855, 1854, 1853,
 1853, 1852, 1851, 1851, 1850, 1849, 1848, 1848,
 1847, 1846, 1846, 1845, 1844, 1844, 1843, 1842,
 1841, 1841, 1840, 1839, 1839, 1838, 1837, 1837,
 1836, 1835, 1834, 1834, 1833, 1832, 1832, 1831,
 1830, 1829, 1829, 1828, 1827, 1826, 1826, 1825,
 1824, 1824, 1823, 1822, 1821, 1821, 1820, 1819,
 1818, 1818, 1817, 1816, 1815, 1815, 1814, 1813,
 1812, 1812, 1811, 1810, 1809, 1809, 1808, 1807,
 1806, 1806, 1805, 1804, 1803, 1803, 1802, 1801,
 1800, 1800, 1799, 1798, 1797, 1796, 1796, 1795,
 1794, 1793, 1793, 1792, 1791, 1790, 1790, 1789,
 1788, 1787, 1786, 1786, 1785, 1784, 1783, 1782,
 1782, 1781, 1780, 1779, 1778, 1778, 1777, 1776,
 1775, 1774, 1774, 1773, 1772, 1771, 1770, 1770,
 1769, 1768, 1767, 1766, 1766, 1765, 1764, 1763,
 1762, 1761, 1761, 1760, 1759, 1758, 1757, 1757,
 1756, 1755, 1754, 1753, 1752, 1752, 1751, 1750,
 1749, 1748, 1747, 1747, 1746, 1745, 1744, 1743,
 1742, 1741, 1741, 1740, 1739, 1738, 1737, 1736,
 1736, 1735, 1734, 1733, 1732, 1731, 1730, 1730,
 1729, 1728, 1727, 1726, 1725, 1724, 1723, 1723,
 1722, 1721, 1720, 1719, 1718, 1717, 1717, 1716,
 1715, 1714, 1713, 1712, 1711, 1710, 1709, 1709,
 1708, 1707, 1706, 1705, 1704, 1703, 1702, 1701,
 1701, 1700, 1699, 1698, 1697, 1696, 1695, 1694,
 1693, 1692, 1692, 1691, 1690, 1689, 1688, 1687,
 1686, 1685, 1684, 1683, 1682, 1682, 1681, 1680,
 1679, 1678, 1677, 1676, 1675, 1674, 1673, 1672,
 1671, 1670, 1670, 1669, 1668, 1667, 1666, 1665,
 1664, 1663, 1662, 1661, 1660, 1659, 1658, 1657,
 1656, 1655, 1654, 1654, 1653, 1652, 1651, 1650,
 1649, 1648, 1647, 1646, 1645, 1644, 1643, 1642,
 1641, 1640, 1639, 1638, 1637, 1636, 1635, 1634,
 1633, 1632, 1631, 1630, 1629, 1629, 1628, 1627,
 1626, 1625, 1624, 1623, 1622, 1621, 1620, 1619,
 1618, 1617, 1616, 1615, 1614, 1613, 1612, 1611,
 1610, 1609, 1608, 1607, 1606, 1605, 1604, 1603,
 1602, 1601, 1600, 1599, 1598, 1597, 1596, 1595,
 1594, 1593, 1592, 1591, 1590, 1589, 1588, 1587,
 1586, 1585, 1584, 1583, 1582, 1581, 1580, 1579,
 1578, 1576, 1575, 1574, 1573, 1572, 1571, 1570,
 1569, 1568, 1567, 1566, 1565, 1564, 1563, 1562,
 1561, 1560, 1559, 1558, 1557, 1556, 1555, 1554,
 1553, 1552, 1551, 1549, 1548, 1547, 1546, 1545,
 1544, 1543, 1542, 1541, 1540, 1539, 1538, 1537,
 1536, 1535, 1534, 1533, 1531, 1530, 1529, 1528,
 1527, 1526, 1525, 1524, 1523, 1522, 1521, 1520,
 1519, 1518, 1516, 1515, 1514, 1513, 1512, 1511,
 1510, 1509, 1508, 1507, 1506, 1504, 1503, 1502,
 1501, 1500, 1499, 1498, 1497, 1496, 1495, 1494,
 1492, 1491, 1490, 1489, 1488, 1487, 1486, 1485,
 1484, 1482, 1481, 1480, 1479, 1478, 1477, 1476,
 1475, 1474, 1472, 1471, 1470, 1469, 1468, 1467,
 1466, 1465, 1463, 1462, 1461, 1460, 1459, 1458,
 1457, 1456, 1454, 1453, 1452, 1451, 1450, 1449,
 1448, 1447, 1445, 1444, 1443, 1442, 1441, 1440,
 1439, 1437, 1436, 1435, 1434, 1433, 1432, 1430,
 1429, 1428, 1427, 1426, 1425, 1424, 1422, 1421,
 1420, 1419, 1418, 1417, 1415, 1414, 1413, 1412,
 1411, 1410, 1408, 1407, 1406, 1405, 1404, 1403,
 1401, 1400, 1399, 1398, 1397, 1396, 1394, 1393,
 1392, 1391, 1390, 1389, 1387, 1386, 1385, 1384,
 1383, 1381, 1380, 1379, 1378, 1377, 1375, 1374,
 1373, 1372, 1371, 1369, 1368, 1367, 1366, 1365,
 1363, 1362, 1361, 1360, 1359, 1357, 1356, 1355,
 1354, 1353, 1351, 1350, 1349, 1348, 1347, 1345,
 1344, 1343, 1342, 1341, 1339, 1338, 1337, 1336,
 1334, 1333, 1332, 1331, 1330, 1328, 1327, 1326,
 1325, 1323, 1322, 1321, 1320, 1319, 1317, 1316,
 1315, 1314, 1312, 1311, 1310, 1309, 1307, 1306,
 1305, 1304, 1302, 1301, 1300, 1299, 1297, 1296,
 1295, 1294, 1293, 1291, 1290, 1289, 1288, 1286,
 1285, 1284, 1283, 1281, 1280, 1279, 1277, 1276,
 1275, 1274, 1272, 1271, 1270, 1269, 1267, 1266,
 1265, 1264, 1262, 1261, 1260, 1259, 1257, 1256,
 1255, 1253, 1252, 1251, 1250, 1248, 1247, 1246,
 1245, 1243, 1242, 1241, 1239, 1238, 1237, 1236,
 1234, 1233, 1232, 1230, 1229, 1228, 1227, 1225,
 1224, 1223, 1221, 1220, 1219, 1218, 1216, 1215,
 1214, 1212, 1211, 1210, 1208, 1207, 1206, 1205,
 1203, 1202, 1201, 1199, 1198, 1197, 1195, 1194,
 1193, 1192, 1190, 1189, 1188, 1186, 1185, 1184,
 1182, 1181, 1180, 1178, 1177, 1176, 1174, 1173,
 1172, 1171, 1169, 1168, 1167, 1165, 1164, 1163,
 1161, 1160, 1159, 1157, 1156, 1155, 1153, 1152,
 1151, 1149, 1148, 1147, 1145, 1144, 1143, 1141,
 1140, 1139, 1137, 1136, 1135, 1133, 1132, 1131,
 1129, 1128, 1127, 1125, 1124, 1123, 1121, 1120,
 1119, 1117, 1116, 1114, 1113, 1112, 1110, 1109,
 1108, 1106, 1105, 1104, 1102, 1101, 1100, 1098,
 1097, 1096, 1094, 1093, 1091, 1090, 1089, 1087,
 1086, 1085, 1083, 1082, 1081, 1079, 1078, 1076,
 1075, 1074, 1072, 1071, 1070, 1068, 1067, 1065,
 1064, 1063, 1061, 1060, 1059, 1057, 1056, 1054,
 1053, 1052, 1050, 1049, 1048, 1046, 1045, 1043,
 1042, 1041, 1039, 1038, 1036, 1035, 1034, 1032,
 1031, 1030, 1028, 1027, 1025, 1024, 1023, 1021,
 1020, 1018, 1017, 1016, 1014, 1013, 1011, 1010,
 1009, 1007, 1006, 1004, 1003, 1002, 1000, 999,
 997, 996, 995, 993, 992, 990, 989, 988,
 986, 985, 983, 982, 980, 979, 978, 976,
 975, 973, 972, 971, 969, 968, 966, 965,
 964, 962, 961, 959, 958, 956, 955, 954,
 952, 951, 949, 948, 946, 945, 944, 942,
 941, 939, 938, 936, 935, 934, 932, 931,
 929, 928, 926, 925, 924, 922, 921, 919,
 918, 916, 915, 913, 912, 911, 909, 908,
 906, 905, 903, 902, 900, 899, 898, 896,
 895, 893, 892, 890, 889, 887, 886, 885,
 883, 882, 880, 879, 877, 876, 874, 873,
 871, 870, 869, 867, 866, 864, 863, 861,
 860, 858, 857, 855, 854, 853, 851, 850,
 848, 847, 845, 844, 842, 841, 839, 838,
 836, 835, 833, 832, 831, 829, 828, 826,
 825, 823, 822, 820, 819, 817, 816, 814,
 813, 811, 810, 808, 807, 805, 804, 803,
 801, 800, 798, 797, 795, 794, 792, 791,
 789, 788, 786, 785, 783, 782, 780, 779,
 777, 776, 774, 773, 771, 770, 768, 767,
 765, 764, 762, 761, 759, 758, 756, 755,
 753, 752, 750, 749, 747, 746, 744, 743,
 741, 740, 738, 737, 735, 734, 732, 731,
 729, 728, 726, 725, 723, 722, 720, 719,
 717, 716, 714, 713, 711, 710, 708, 707,
 705, 704, 702, 701, 699, 698, 696, 695,
 693, 692, 690, 689, 687, 686, 684, 683,
 681, 680, 678, 677, 675, 674, 672, 670,
 669, 667, 666, 664, 663, 661, 660, 658,
 657, 655, 654, 652, 651, 649, 648, 646,
 645, 643, 642, 640, 638, 637, 635, 634,
 632, 631, 629, 628, 626, 625, 623, 622,
 620, 619, 617, 616, 614, 612, 611, 609,
 608, 606, 605, 603, 602, 600, 599, 597,
 596, 594, 592, 591, 589, 588, 586, 585,
 583, 582, 580, 579, 577, 576, 574, 572,
 571, 569, 568, 566, 565, 563, 562, 560,
 559, 557, 555, 554, 552, 551, 549, 548,
 546, 545, 543, 541, 540, 538, 537, 535,
 534, 532, 531, 529, 527, 526, 524, 523,
 521, 520, 518, 517, 515, 513, 512, 510,
 509, 507, 506, 504, 503, 501, 499, 498,
 496, 495, 493, 492, 490, 489, 487, 485,
 484, 482, 481, 479, 478, 476, 474, 473,
 471, 470, 468, 467, 465, 464, 462, 460,
 459, 457, 456, 454, 453, 451, 449, 448,
 446, 445, 443, 442, 440, 438, 437, 435,
 434, 432, 431, 429, 427, 426, 424, 423,
 421, 420, 418, 416, 415, 413, 412, 410,
 409, 407, 405, 404, 402, 401, 399, 397,
 396, 394, 393, 391, 390, 388, 386, 385,
 383, 382, 380, 379, 377, 375, 374, 372,
 371, 369, 367, 366, 364, 363, 361, 360,
 358, 356, 355, 353, 352, 350, 348, 347,
 345, 344, 342, 341, 339, 337, 336, 334,
 333, 331, 329, 328, 326, 325, 323, 321,
 320, 318, 317, 315, 314, 312, 310, 309,
 307, 306, 304, 302, 301, 299, 298, 296,
 294, 293, 291, 290, 288, 286, 285, 283,
 282, 280, 279, 277, 275, 274, 272, 271,
 269, 267, 266, 264, 263, 261, 259, 258,
 256, 255, 253, 251, 250, 248, 247, 245,
 243, 242, 240, 239, 237, 235, 234, 232,
 231, 229, 227, 226, 224, 223, 221, 219,
 218, 216, 215, 213, 211, 210, 208, 207,
 205, 203, 202, 200, 199, 197, 195, 194,
 192, 191, 189, 187, 186, 184, 183, 181,
 179, 178, 176, 175, 173, 171, 170, 168,
 167, 165, 163, 162, 160, 159, 157, 155,
 154, 152, 151, 149, 147, 146, 144, 143,
 141, 139, 138, 136, 135, 133, 131, 130,
 128, 126, 125, 123, 122, 120, 118, 117,
 115, 114, 112, 110, 109, 107, 106, 104,
 102, 101, 99, 98, 96, 94, 93, 91,
 90, 88, 86, 85, 83, 82, 80, 78,
 77, 75, 73, 72, 70, 69, 67, 65,
 64, 62, 61, 59, 57, 56, 54, 53,
 51, 49, 48, 46, 45, 43, 41, 40,
 38, 36, 35, 33, 32, 30, 28, 27,
 25, 24, 22, 20, 19, 17, 16, 14,
 12, 11, 9, 8, 6, 4, 3, 1,
 0, -1, -3, -4, -6, -8, -9, -11,
 -12, -14, -16, -17, -19, -20, -22, -24,
 -25, -27, -28, -30, -32, -33, -35, -36,
 -38, -40, -41, -43, -45, -46, -48, -49,
 -51, -53, -54, -56, -57, -59, -61, -62,
 -64, -65, -67, -69, -70, -72, -73, -75,
 -77, -78, -80, -82, -83, -85, -86, -88,
 -90, -91, -93, -94, -96, -98, -99, -101,
 -102, -104, -106, -107, -109, -110, -112, -114,
 -115, -117, -118, -120, -122, -123, -125, -126,
 -128, -130, -131, -133, -135, -136, -138, -139,
 -141, -143, -144, -146, -147, -149, -151, -152,
 -154, -155, -157, -159, -160, -162, -163, -165,
 -167, -168, -170, -171, -173, -175, -176, -178,
 -179, -181, -183, -184, -186, -187, -189, -191,
 -192, -194, -195, -197, -199, -200, -202, -203,
 -205, -207, -208, -210, -211, -213, -215, -216,
 -218, -219, -221, -223, -224, -226, -227, -229,
 -231, -232, -234, -235, -237, -239, -240, -242,
 -243, -245, -247, -248, -250, -251, -253, -255,
 -256, -258, -259, -261, -263, -264, -266, -267,
 -269, -271, -272, -274, -275, -277, -279, -280,
 -282, -283, -285, -286, -288, -290, -291, -293,
 -294, -296, -298, -299, -301, -302, -304, -306,
 -307, -309, -310, -312, -314, -315, -317, -318,
 -320, -321, -323, -325, -326, -328, -329, -331,
 -333, -334, -336, -337, -339, -341, -342, -344,
 -345, -347, -348, -350, -352, -353, -355, -356,
 -358, -360, -361, -363, -364, -366, -367, -369,
 -371, -372, -374, -375, -377, -379, -380, -382,
 -383, -385, -386, -388, -390, -391, -393, -394,
 -396, -397, -399, -401, -402, -404, -405, -407,
 -409, -410, -412, -413, -415, -416, -418, -420,
 -421, -423, -424, -426, -427, -429, -431, -432,
 -434, -435, -437, -438, -440, -442, -443, -445,
 -446, -448, -449, -451, -453, -454, -456, -457,
 -459, -460, -462, -464, -465, -467, -468, -470,
 -471, -473, -474, -476, -478, -479, -481, -482,
 -484, -485, -487, -489, -490, -492, -493, -495,
 -496, -498, -499, -501, -503, -504, -506, -507,
 -509, -510, -512, -513, -515, -517, -518, -520,
 -521, -523, -524, -526, -527, -529, -531, -532,
 -534, -535, -537, -538, -540, -541, -543, -545,
 -546, -548, -549, -551, -552, -554, -555, -557,
 -559, -560, -562, -563, -565, -566, -568, -569,
 -571, -572, -574, -576, -577, -579, -580, -582,
 -583, -585, -586, -588, -589, -591, -592, -594,
 -596, -597, -599, -600, -602, -603, -605, -606,
 -608, -609, -611, -612, -614, -616, -617, -619,
 -620, -622, -623, -625, -626, -628, -629, -631,
 -632, -634, -635, -637, -638, -640, -642, -643,
 -645, -646, -648, -649, -651, -652, -654, -655,
 -657, -658, -660, -661, -663, -664, -666, -667,
 -669, -670, -672, -674, -675, -677, -678, -680,
 -681, -683, -684, -686, -687, -689, -690, -692,
 -693, -695, -696, -698, -699, -701, -702, -704,
 -705, -707, -708, -710, -711, -713, -714, -716,
 -717, -719, -720, -722, -723, -725, -726, -728,
 -729, -731, -732, -734, -735, -737, -738, -740,
 -741, -743, -744, -746, -747, -749, -750, -752,
 -753, -755, -756, -758, -759, -761, -762, -764,
 -765, -767, -768, -770, -771, -773, -774, -776,
 -777, -779, -780, -782, -783, -785, -786, -788,
 -789, -791, -792, -794, -795, -797, -798, -800,
 -801, -803, -804, -805, -807, -808, -810, -811,
 -813, -814, -816, -817, -819, -820, -822, -823,
 -825, -826, -828, -829, -831, -832, -833, -835,
 -836, -838, -839, -841, -842, -844, -845, -847,
 -848, -850, -851, -853, -854, -855, -857, -858,
 -860, -861, -863, -864, -866, -867, -869, -870,
 -871, -873, -874, -876, -877, -879, -880, -882,
 -883, -885, -886, -887, -889, -890, -892, -893,
 -895, -896, -898, -899, -900, -902, -903, -905,
 -906, -908, -909, -911, -912, -913, -915, -916,
 -918, -919, -921, -922, -924, -925, -926, -928,
 -929, -931, -932, -934, -935, -936, -938, -939,
 -941, -942, -944, -945, -946, -948, -949, -951,
 -952, -954, -955, -956, -958, -959, -961, -962,
 -964, -965, -966, -968, -969, -971, -972, -973,
 -975, -976, -978, -979, -980, -982, -983, -985,
 -986, -988, -989, -990, -992, -993, -995, -996,
 -997, -999, -1000, -1002, -1003, -1004, -1006, -1007,
 -1009, -1010, -1011, -1013, -1014, -1016, -1017, -1018,
 -1020, -1021, -1023, -1024, -1025, -1027, -1028, -1030,
 -1031, -1032, -1034, -1035, -1036, -1038, -1039, -1041,
 -1042, -1043, -1045, -1046, -1048, -1049, -1050, -1052,
 -1053, -1054, -1056, -1057, -1059, -1060, -1061, -1063,
 -1064, -1065, -1067, -1068, -1070, -1071, -1072, -1074,
 -1075, -1076, -1078, -1079, -1081, -1082, -1083, -1085,
 -1086, -1087, -1089, -1090, -1091, -1093, -1094, -1096,
 -1097, -1098, -1100, -1101, -1102, -1104, -1105, -1106,
 -1108, -1109, -1110, -1112, -1113, -1114, -1116, -1117,
 -1119, -1120, -1121, -1123, -1124, -1125, -1127, -1128,
 -1129, -1131, -1132, -1133, -1135, -1136, -1137, -1139,
 -1140, -1141, -1143, -1144, -1145, -1147, -1148, -1149,
 -1151, -1152, -1153, -1155, -1156, -1157, -1159, -1160,
 -1161, -1163, -1164, -1165, -1167, -1168, -1169, -1171,
 -1172, -1173, -1174, -1176, -1177, -1178, -1180, -1181,
 -1182, -1184, -1185, -1186, -1188, -1189, -1190, -1192,
 -1193, -1194, -1195, -1197, -1198, -1199, -1201, -1202,
 -1203, -1205, -1206, -1207, -1208, -1210, -1211, -1212,
 -1214, -1215, -1216, -1218, -1219, -1220, -1221, -1223,
 -1224, -1225, -1227, -1228, -1229, -1230, -1232, -1233,
 -1234, -1236, -1237, -1238, -1239, -1241, -1242, -1243,
 -1245, -1246, -1247, -1248, -1250, -1251, -1252, -1253,
 -1255, -1256, -1257, -1259, -1260, -1261, -1262, -1264,
 -1265, -1266, -1267, -1269, -1270, -1271, -1272, -1274,
 -1275, -1276, -1277, -1279, -1280, -1281, -1283, -1284,
 -1285, -1286, -1288, -1289, -1290, -1291, -1293, -1294,
 -1295, -1296, -1297, -1299, -1300, -1301, -1302, -1304,
 -1305, -1306, -1307, -1309, -1310, -1311, -1312, -1314,
 -1315, -1316, -1317, -1319, -1320, -1321, -1322, -1323,
 -1325, -1326, -1327, -1328, -1330, -1331, -1332, -1333,
 -1334, -1336, -1337, -1338, -1339, -1341, -1342, -1343,
 -1344, -1345, -1347, -1348, -1349, -1350, -1351, -1353,
 -1354, -1355, -1356, -1357, -1359, -1360, -1361, -1362,
 -1363, -1365, -1366, -1367, -1368, -1369, -1371, -1372,
 -1373, -1374, -1375, -1377, -1378, -1379, -1380, -1381,
 -1383, -1384, -1385, -1386, -1387, -1389, -1390, -1391,
 -1392, -1393, -1394, -1396, -1397, -1398, -1399, -1400,
 -1401, -1403, -1404, -1405, -1406, -1407, -1408, -1410,
 -1411, -1412, -1413, -1414, -1415, -1417, -1418, -1419,
 -1420, -1421, -1422, -1424, -1425, -1426, -1427, -1428,
 -1429, -1430, -1432, -1433, -1434, -1435, -1436, -1437,
 -1439, -1440, -1441, -1442, -1443, -1444, -1445, -1447,
 -1448, -1449, -1450, -1451, -1452, -1453, -1454, -1456,
 -1457, -1458, -1459, -1460, -1461, -1462, -1463, -1465,
 -1466, -1467, -1468, -1469, -1470, -1471, -1472, -1474,
 -1475, -1476, -1477, -1478, -1479, -1480, -1481, -1482,
 -1484, -1485, -1486, -1487, -1488, -1489, -1490, -1491,
 -1492, -1494, -1495, -1496, -1497, -1498, -1499, -1500,
 -1501, -1502, -1503, -1504, -1506, -1507, -1508, -1509,
 -1510, -1511, -1512, -1513, -1514, -1515, -1516, -1518,
 -1519, -1520, -1521, -1522, -1523, -1524, -1525, -1526,
 -1527, -1528, -1529, -1530, -1531, -1533, -1534, -1535,
 -1536, -1537, -1538, -1539, -1540, -1541, -1542, -1543,
 -1544, -1545, -1546, -1547, -1548, -1549, -1551, -1552,
 -1553, -1554, -1555, -1556, -1557, -1558, -1559, -1560,
 -1561, -1562, -1563, -1564, -1565, -1566, -1567, -1568,
 -1569, -1570, -1571, -1572, -1573, -1574, -1575, -1576,
 -1578, -1579, -1580, -1581, -1582, -1583, -1584, -1585,
 -1586, -1587, -1588, -1589, -1590, -1591, -1592, -1593,
 -1594, -1595, -1596, -1597, -1598, -1599, -1600, -1601,
 -1602, -1603, -1604, -1605, -1606, -1607, -1608, -1609,
 -1610, -1611, -1612, -1613, -1614, -1615, -1616, -1617,
 -1618, -1619, -1620, -1621, -1622, -1623, -1624, -1625,
 -1626, -1627, -1628, -1629, -1629, -1630, -1631, -1632,
 -1633, -1634, -1635, -1636, -1637, -1638, -1639, -1640,
 -1641, -1642, -1643, -1644, -1645, -1646, -1647, -1648,
 -1649, -1650, -1651, -1652, -1653, -1654, -1654, -1655,
 -1656, -1657, -1658, -1659, -1660, -1661, -1662, -1663,
 -1664, -1665, -1666, -1667, -1668, -1669, -1670, -1670,
 -1671, -1672, -1673, -1674, -1675, -1676, -1677, -1678,
 -1679, -1680, -1681, -1682, -1682, -1683, -1684, -1685,
 -1686, -1687, -1688, -1689, -1690, -1691, -1692, -1692,
 -1693, -1694, -1695, -1696, -1697, -1698, -1699, -1700,
 -1701, -1701, -1702, -1703, -1704, -1705, -1706, -1707,
 -1708, -1709, -1709, -1710, -1711, -1712, -1713, -1714,
 -1715, -1716, -1717, -1717, -1718, -1719, -1720, -1721,
 -1722, -1723, -1723, -1724, -1725, -1726, -1727, -1728,
 -1729, -1730, -1730, -1731, -1732, -1733, -1734, -1735,
 -1736, -1736, -1737, -1738, -1739, -1740, -1741, -1741,
 -1742, -1743, -1744, -1745, -1746, -1747, -1747, -1748,
 -1749, -1750, -1751, -1752, -1752, -1753, -1754, -1755,
 -1756, -1757, -1757, -1758, -1759, -1760, -1761, -1761,
 -1762, -1763, -1764, -1765, -1766, -1766, -1767, -1768,
 -1769, -1770, -1770, -1771, -1772, -1773, -1774, -1774,
 -1775, -1776, -1777, -1778, -1778, -1779, -1780, -1781,
 -1782, -1782, -1783, -1784, -1785, -1786, -1786, -1787,
 -1788, -1789, -1790, -1790, -1791, -1792, -1793, -1793,
 -1794, -1795, -1796, -1796, -1797, -1798, -1799, -1800,
 -1800, -1801, -1802, -1803, -1803, -1804, -1805, -1806,
 -1806, -1807, -1808, -1809, -1809, -1810, -1811, -1812,
 -1812, -1813, -1814, -1815, -1815, -1816, -1817, -1818,
 -1818, -1819, -1820, -1821, -1821, -1822, -1823, -1824,
 -1824, -1825, -1826, -1826, -1827, -1828, -1829, -1829,
 -1830, -1831, -1832, -1832, -1833, -1834, -1834, -1835,
 -1836, -1837, -1837, -1838, -1839, -1839, -1840, -1841,
 -1841, -1842, -1843, -1844, -1844, -1845, -1846, -1846,
 -1847, -1848, -1848, -1849, -1850, -1851, -1851, -1852,
 -1853, -1853, -1854, -1855, -1855, -1856, -1857, -1857,
 -1858, -1859, -1859, -1860, -1861, -1861, -1862, -1863,
 -1863, -1864, -1865, -1865, -1866, -1867, -1867, -1868,
 -1869, -1869, -1870, -1871, -1871, -1872, -1873, -1873,
 -1874, -1875, -1875, -1876, -1876, -1877, -1878, -1878,
 -1879, -1880, -1880, -1881, -1882, -1882, -1883, -1884,
 -1884, -1885, -1885, -1886, -1887, -1887, -1888, -1889,
 -1889, -1890, -1890, -1891, -1892, -1892, -1893, -1893,
 -1894, -1895, -1895, -1896, -1896, -1897, -1898, -1898,
 -1899, -1900, -1900, -1901, -1901, -1902, -1902, -1903,
 -1904, -1904, -1905, -1905, -1906, -1907, -1907, -1908,
 -1908, -1909, -1910, -1910, -1911, -1911, -1912, -1912,
 -1913, -1914, -1914, -1915, -1915, -1916, -1916, -1917,
 -1918, -1918, -1919, -1919, -1920, -1920, -1921, -1921,
 -1922, -1923, -1923, -1924, -1924, -1925, -1925, -1926,
 -1926, -1927, -1928, -1928, -1929, -1929, -1930, -1930,
 -1931, -1931, -1932, -1932, -1933, -1933, -1934, -1934,
 -1935, -1936, -1936, -1937, -1937, -1938, -1938, -1939,
 -1939, -1940, -1940, -1941, -1941, -1942, -1942, -1943,
 -1943, -1944, -1944, -1945, -1945, -1946, -1946, -1947,
 -1947, -1948, -1948, -1949, -1949, -1950, -1950, -1951,
 -1951, -1952, -1952, -1953, -1953, -1954, -1954, -1955,
 -1955, -1956, -1956, -1956, -1957, -1957, -1958, -1958,
 -1959, -1959, -1960, -1960, -1961, -1961, -1962, -1962,
 -1963, -1963, -1963, -1964, -1964, -1965, -1965, -1966,
 -1966, -1967, -1967, -1968, -1968, -1968, -1969, -1969,
 -1970, -1970, -1971, -1971, -1971, -1972, -1972, -1973,
 -1973, -1974, -1974, -1974, -1975, -1975, -1976, -1976,
 -1977, -1977, -1977, -1978, -1978, -1979, -1979, -1980,
 -1980, -1980, -1981, -1981, -1982, -1982, -1982, -1983,
 -1983, -1984, -1984, -1984, -1985, -1985, -1986, -1986,
 -1986, -1987, -1987, -1987, -1988, -1988, -1989, -1989,
 -1989, -1990, -1990, -1991, -1991, -1991, -1992, -1992,
 -1992, -1993, -1993, -1994, -1994, -1994, -1995, -1995,
 -1995, -1996, -1996, -1996, -1997, -1997, -1997, -1998,
 -1998, -1999, -1999, -1999, -2000, -2000, -2000, -2001,
 -2001, -2001, -2002, -2002, -2002, -2003, -2003, -2003,
 -2004, -2004, -2004, -2005, -2005, -2005, -2006, -2006,
 -2006, -2007, -2007, -2007, -2008, -2008, -2008, -2008,
 -2009, -2009, -2009, -2010, -2010, -2010, -2011, -2011,
 -2011, -2012, -2012, -2012, -2012, -2013, -2013, -2013,
 -2014, -2014, -2014, -2014, -2015, -2015, -2015, -2016,
 -2016, -2016, -2016, -2017, -2017, -2017, -2018, -2018,
 -2018, -2018, -2019, -2019, -2019, -2019, -2020, -2020,
 -2020, -2020, -2021, -2021, -2021, -2022, -2022, -2022,
 -2022, -2023, -2023, -2023, -2023, -2024, -2024, -2024,
 -2024, -2024, -2025, -2025, -2025, -2025, -2026, -2026,
 -2026, -2026, -2027, -2027, -2027, -2027, -2028, -2028,
 -2028, -2028, -2028, -2029, -2029, -2029, -2029, -2029,
 -2030, -2030, -2030, -2030, -2031, -2031, -2031, -2031,
 -2031, -2032, -2032, -2032, -2032, -2032, -2033, -2033,
 -2033, -2033, -2033, -2033, -2034, -2034, -2034, -2034,
 -2034, -2035, -2035, -2035, -2035, -2035, -2035, -2036,
 -2036, -2036, -2036, -2036, -2037, -2037, -2037, -2037,
 -2037, -2037, -2037, -2038, -2038, -2038, -2038, -2038,
 -2038, -2039, -2039, -2039, -2039, -2039, -2039, -2039,
 -2040, -2040, -2040, -2040, -2040, -2040, -2040, -2041,
 -2041, -2041, -2041, -2041, -2041, -2041, -2041, -2042,
 -2042, -2042, -2042, -2042, -2042, -2042, -2042, -2042,
 -2043, -2043, -2043, -2043, -2043, -2043, -2043, -2043,
 -2043, -2044, -2044, -2044, -2044, -2044, -2044, -2044,
 -2044, -2044, -2044, -2044, -2045, -2045, -2045, -2045,
 -2045, -2045, -2045, -2045, -2045, -2045, -2045, -2045,
 -2046, -2046, -2046, -2046, -2046, -2046, -2046, -2046,
 -2046, -2046, -2046, -2046, -2046, -2046, -2046, -2046,
 -2046, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2048, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2047, -2047, -2047, -2047, -2047, -2047, -2047, -2047,
 -2046, -2046, -2046, -2046, -2046, -2046, -2046, -2046,
 -2046, -2046, -2046, -2046, -2046, -2046, -2046, -2046,
 -2046, -2045, -2045, -2045, -2045, -2045, -2045, -2045,
 -2045, -2045, -2045, -2045, -2045, -2044, -2044, -2044,
 -2044, -2044, -2044, -2044, -2044, -2044, -2044, -2044,
 -2043, -2043, -2043, -2043, -2043, -2043, -2043, -2043,
 -2043, -2042, -2042, -2042, -2042, -2042, -2042, -2042,
 -2042, -2042, -2041, -2041, -2041, -2041, -2041, -2041,
 -2041, -2041, -2040, -2040, -2040, -2040, -2040, -2040,
 -2040, -2039, -2039, -2039, -2039, -2039, -2039, -2039,
 -2038, -2038, -2038, -2038, -2038, -2038, -2037, -2037,
 -2037, -2037, -2037, -2037, -2037, -2036, -2036, -2036,
 -2036, -2036, -2035, -2035, -2035, -2035, -2035, -2035,
 -2034, -2034, -2034, -2034, -2034, -2033, -2033, -2033,
 -2033, -2033, -2033, -2032, -2032, -2032, -2032, -2032,
 -2031, -2031, -2031, -2031, -2031, -2030, -2030, -2030,
 -2030, -2029, -2029, -2029, -2029, -2029, -2028, -2028,
 -2028, -2028, -2028, -2027, -2027, -2027, -2027, -2026,
 -2026, -2026, -2026, -2025, -2025, -2025, -2025, -2024,
 -2024, -2024, -2024, -2024, -2023, -2023, -2023, -2023,
 -2022, -2022, -2022, -2022, -2021, -2021, -2021, -2020,
 -2020, -2020, -2020, -2019, -2019, -2019, -2019, -2018,
 -2018, -2018, -2018, -2017, -2017, -2017, -2016, -2016,
 -2016, -2016, -2015, -2015, -2015, -2014, -2014, -2014,
 -2014, -2013, -2013, -2013, -2012, -2012, -2012, -2012,
 -2011, -2011, -2011, -2010, -2010, -2010, -2009, -2009,
 -2009, -2008, -2008, -2008, -2008, -2007, -2007, -2007,
 -2006, -2006, -2006, -2005, -2005, -2005, -2004, -2004,
 -2004, -2003, -2003, -2003, -2002, -2002, -2002, -2001,
 -2001, -2001, -2000, -2000, -2000, -1999, -1999, -1999,
 -1998, -1998, -1997, -1997, -1997, -1996, -1996, -1996,
 -1995, -1995, -1995, -1994, -1994, -1994, -1993, -1993,
 -1992, -1992, -1992, -1991, -1991, -1991, -1990, -1990,
 -1989, -1989, -1989, -1988, -1988, -1987, -1987, -1987,
 -1986, -1986, -1986, -1985, -1985, -1984, -1984, -1984,
 -1983, -1983, -1982, -1982, -1982, -1981, -1981, -1980,
 -1980, -1980, -1979, -1979, -1978, -1978, -1977, -1977,
 -1977, -1976, -1976, -1975, -1975, -1974, -1974, -1974,
 -1973, -1973, -1972, -1972, -1971, -1971, -1971, -1970,
 -1970, -1969, -1969, -1968, -1968, -1968, -1967, -1967,
 -1966, -1966, -1965, -1965, -1964, -1964, -1963, -1963,
 -1963, -1962, -1962, -1961, -1961, -1960, -1960, -1959,
 -1959, -1958, -1958, -1957, -1957, -1956, -1956, -1956,
 -1955, -1955, -1954, -1954, -1953, -1953, -1952, -1952,
 -1951, -1951, -1950, -1950, -1949, -1949, -1948, -1948,
 -1947, -1947, -1946, -1946, -1945, -1945, -1944, -1944,
 -1943, -1943, -1942, -1942, -1941, -1941, -1940, -1940,
 -1939, -1939, -1938, -1938, -1937, -1937, -1936, -1936,
 -1935, -1934, -1934, -1933, -1933, -1932, -1932, -1931,
 -1931, -1930, -1930, -1929, -1929, -1928, -1928, -1927,
 -1926, -1926, -1925, -1925, -1924, -1924, -1923, -1923,
 -1922, -1921, -1921, -1920, -1920, -1919, -1919, -1918,
 -1918, -1917, -1916, -1916, -1915, -1915, -1914, -1914,
 -1913, -1912, -1912, -1911, -1911, -1910, -1910, -1909,
 -1908, -1908, -1907, -1907, -1906, -1905, -1905, -1904,
 -1904, -1903, -1902, -1902, -1901, -1901, -1900, -1900,
 -1899, -1898, -1898, -1897, -1896, -1896, -1895, -1895,
 -1894, -1893, -1893, -1892, -1892, -1891, -1890, -1890,
 -1889, -1889, -1888, -1887, -1887, -1886, -1885, -1885,
 -1884, -1884, -1883, -1882, -1882, -1881, -1880, -1880,
 -1879, -1878, -1878, -1877, -1876, -1876, -1875, -1875,
 -1874, -1873, -1873, -1872, -1871, -1871, -1870, -1869,
 -1869, -1868, -1867, -1867, -1866, -1865, -1865, -1864,
 -1863, -1863, -1862, -1861, -1861, -1860, -1859, -1859,
 -1858, -1857, -1857, -1856, -1855, -1855, -1854, -1853,
 -1853, -1852, -1851, -1851, -1850, -1849, -1848, -1848,
 -1847, -1846, -1846, -1845, -1844, -1844, -1843, -1842,
 -1841, -1841, -1840, -1839, -1839, -1838, -1837, -1837,
 -1836, -1835, -1834, -1834, -1833, -1832, -1832, -1831,
 -1830, -1829, -1829, -1828, -1827, -1826, -1826, -1825,
 -1824, -1824, -1823, -1822, -1821, -1821, -1820, -1819,
 -1818, -1818, -1817, -1816, -1815, -1815, -1814, -1813,
 -1812, -1812, -1811, -1810, -1809, -1809, -1808, -1807,
 -1806, -1806, -1805, -1804, -1803, -1803, -1802, -1801,
 -1800, -1800, -1799, -1798, -1797, -1796, -1796, -1795,
 -1794, -1793, -1793, -1792, -1791, -1790, -1790, -1789,
 -1788, -1787, -1786, -1786, -1785, -1784, -1783, -1782,
 -1782, -1781, -1780, -1779, -1778, -1778, -1777, -1776,
 -1775, -1774, -1774, -1773, -1772, -1771, -1770, -1770,
 -1769, -1768, -1767, -1766, -1766, -1765, -1764, -1763,
 -1762, -1761, -1761, -1760, -1759, -1758, -1757, -1757,
 -1756, -1755, -1754, -1753, -1752, -1752, -1751, -1750,
 -1749, -1748, -1747, -1747, -1746, -1745, -1744, -1743,
 -1742, -1741, -1741, -1740, -1739, -1738, -1737, -1736,
 -1736, -1735, -1734, -1733, -1732, -1731, -1730, -1730,
 -1729, -1728, -1727, -1726, -1725, -1724, -1723, -1723,
 -1722, -1721, -1720, -1719, -1718, -1717, -1717, -1716,
 -1715, -1714, -1713, -1712, -1711, -1710, -1709, -1709,
 -1708, -1707, -1706, -1705, -1704, -1703, -1702, -1701,
 -1701, -1700, -1699, -1698, -1697, -1696, -1695, -1694,
 -1693, -1692, -1692, -1691, -1690, -1689, -1688, -1687,
 -1686, -1685, -1684, -1683, -1682, -1682, -1681, -1680,
 -1679, -1678, -1677, -1676, -1675, -1674, -1673, -1672,
 -1671, -1670, -1670, -1669, -1668, -1667, -1666, -1665,
 -1664, -1663, -1662, -1661, -1660, -1659, -1658, -1657,
 -1656, -1655, -1654, -1654, -1653, -1652, -1651, -1650,
 -1649, -1648, -1647, -1646, -1645, -1644, -1643, -1642,
 -1641, -1640, -1639, -1638, -1637, -1636, -1635, -1634,
 -1633, -1632, -1631, -1630, -1629, -1629, -1628, -1627,
 -1626, -1625, -1624, -1623, -1622, -1621, -1620, -1619,
 -1618, -1617, -1616, -1615, -1614, -1613, -1612, -1611,
 -1610, -1609, -1608, -1607, -1606, -1605, -1604, -1603,
 -1602, -1601, -1600, -1599, -1598, -1597, -1596, -1595,
 -1594, -1593, -1592, -1591, -1590, -1589, -1588, -1587,
 -1586, -1585, -1584, -1583, -1582, -1581, -1580, -1579,
 -1578, -1576, -1575, -1574, -1573, -1572, -1571, -1570,
 -1569, -1568, -1567, -1566, -1565, -1564, -1563, -1562,
 -1561, -1560, -1559, -1558, -1557, -1556, -1555, -1554,
 -1553, -1552, -1551, -1549, -1548, -1547, -1546, -1545,
 -1544, -1543, -1542, -1541, -1540, -1539, -1538, -1537,
 -1536, -1535, -1534, -1533, -1531, -1530, -1529, -1528,
 -1527, -1526, -1525, -1524, -1523, -1522, -1521, -1520,
 -1519, -1518, -1516, -1515, -1514, -1513, -1512, -1511,
 -1510, -1509, -1508, -1507, -1506, -1504, -1503, -1502,
 -1501, -1500, -1499, -1498, -1497, -1496, -1495, -1494,
 -1492, -1491, -1490, -1489, -1488, -1487, -1486, -1485,
 -1484, -1482, -1481, -1480, -1479, -1478, -1477, -1476,
 -1475, -1474, -1472, -1471, -1470, -1469, -1468, -1467,
 -1466, -1465, -1463, -1462, -1461, -1460, -1459, -1458,
 -1457, -1456, -1454, -1453, -1452, -1451, -1450, -1449,
 -1448, -1447, -1445, -1444, -1443, -1442, -1441, -1440,
 -1439, -1437, -1436, -1435, -1434, -1433, -1432, -1430,
 -1429, -1428, -1427, -1426, -1425, -1424, -1422, -1421,
 -1420, -1419, -1418, -1417, -1415, -1414, -1413, -1412,
 -1411, -1410, -1408, -1407, -1406, -1405, -1404, -1403,
 -1401, -1400, -1399, -1398, -1397, -1396, -1394, -1393,
 -1392, -1391, -1390, -1389, -1387, -1386, -1385, -1384,
 -1383, -1381, -1380, -1379, -1378, -1377, -1375, -1374,
 -1373, -1372, -1371, -1369, -1368, -1367, -1366, -1365,
 -1363, -1362, -1361, -1360, -1359, -1357, -1356, -1355,
 -1354, -1353, -1351, -1350, -1349, -1348, -1347, -1345,
 -1344, -1343, -1342, -1341, -1339, -1338, -1337, -1336,
 -1334, -1333, -1332, -1331, -1330, -1328, -1327, -1326,
 -1325, -1323, -1322, -1321, -1320, -1319, -1317, -1316,
 -1315, -1314, -1312, -1311, -1310, -1309, -1307, -1306,
 -1305, -1304, -1302, -1301, -1300, -1299, -1297, -1296,
 -1295, -1294, -1293, -1291, -1290, -1289, -1288, -1286,
 -1285, -1284, -1283, -1281, -1280, -1279, -1277, -1276,
 -1275, -1274, -1272, -1271, -1270, -1269, -1267, -1266,
 -1265, -1264, -1262, -1261, -1260, -1259, -1257, -1256,
 -1255, -1253, -1252, -1251, -1250, -1248, -1247, -1246,
 -1245, -1243, -1242, -1241, -1239, -1238, -1237, -1236,
 -1234, -1233, -1232, -1230, -1229, -1228, -1227, -1225,
 -1224, -1223, -1221, -1220, -1219, -1218, -1216, -1215,
 -1214, -1212, -1211, -1210, -1208, -1207, -1206, -1205,
 -1203, -1202, -1201, -1199, -1198, -1197, -1195, -1194,
 -1193, -1192, -1190, -1189, -1188, -1186, -1185, -1184,
 -1182, -1181, -1180, -1178, -1177, -1176, -1174, -1173,
 -1172, -1171, -1169, -1168, -1167, -1165, -1164, -1163,
 -1161, -1160, -1159, -1157, -1156, -1155, -1153, -1152,
 -1151, -1149, -1148, -1147, -1145, -1144, -1143, -1141,
 -1140, -1139, -1137, -1136, -1135, -1133, -1132, -1131,
 -1129, -1128, -1127, -1125, -1124, -1123, -1121, -1120,
 -1119, -1117, -1116, -1114, -1113, -1112, -1110, -1109,
 -1108, -1106, -1105, -1104, -1102, -1101, -1100, -1098,
 -1097, -1096, -1094, -1093, -1091, -1090, -1089, -1087,
 -1086, -1085, -1083, -1082, -1081, -1079, -1078, -1076,
 -1075, -1074, -1072, -1071, -1070, -1068, -1067, -1065,
 -1064, -1063, -1061, -1060, -1059, -1057, -1056, -1054,
 -1053, -1052, -1050, -1049, -1048, -1046, -1045, -1043,
 -1042, -1041, -1039, -1038, -1036, -1035, -1034, -1032,
 -1031, -1030, -1028, -1027, -1025, -1024, -1023, -1021,
 -1020, -1018, -1017, -1016, -1014, -1013, -1011, -1010,
 -1009, -1007, -1006, -1004, -1003, -1002, -1000, -999,
 -997, -996, -995, -993, -992, -990, -989, -988,
 -986, -985, -983, -982, -980, -979, -978, -976,
 -975, -973, -972, -971, -969, -968, -966, -965,
 -964, -962, -961, -959, -958, -956, -955, -954,
 -952, -951, -949, -948, -946, -945, -944, -942,
 -941, -939, -938, -936, -935, -934, -932, -931,
 -929, -928, -926, -925, -924, -922, -921, -919,
 -918, -916, -915, -913, -912, -911, -909, -908,
 -906, -905, -903, -902, -900, -899, -898, -896,
 -895, -893, -892, -890, -889, -887, -886, -885,
 -883, -882, -880, -879, -877, -876, -874, -873,
 -871, -870, -869, -867, -866, -864, -863, -861,
 -860, -858, -857, -855, -854, -853, -851, -850,
 -848, -847, -845, -844, -842, -841, -839, -838,
 -836, -835, -833, -832, -831, -829, -828, -826,
 -825, -823, -822, -820, -819, -817, -816, -814,
 -813, -811, -810, -808, -807, -805, -804, -803,
 -801, -800, -798, -797, -795, -794, -792, -791,
 -789, -788, -786, -785, -783, -782, -780, -779,
 -777, -776, -774, -773, -771, -770, -768, -767,
 -765, -764, -762, -761, -759, -758, -756, -755,
 -753, -752, -750, -749, -747, -746, -744, -743,
 -741, -740, -738, -737, -735, -734, -732, -731,
 -729, -728, -726, -725, -723, -722, -720, -719,
 -717, -716, -714, -713, -711, -710, -708, -707,
 -705, -704, -702, -701, -699, -698, -696, -695,
 -693, -692, -690, -689, -687, -686, -684, -683,
 -681, -680, -678, -677, -675, -674, -672, -670,
 -669, -667, -666, -664, -663, -661, -660, -658,
 -657, -655, -654, -652, -651, -649, -648, -646,
 -645, -643, -642, -640, -638, -637, -635, -634,
 -632, -631, -629, -628, -626, -625, -623, -622,
 -620, -619, -617, -616, -614, -612, -611, -609,
 -608, -606, -605, -603, -602, -600, -599, -597,
 -596, -594, -592, -591, -589, -588, -586, -585,
 -583, -582, -580, -579, -577, -576, -574, -572,
 -571, -569, -568, -566, -565, -563, -562, -560,
 -559, -557, -555, -554, -552, -551, -549, -548,
 -546, -545, -543, -541, -540, -538, -537, -535,
 -534, -532, -531, -529, -527, -526, -524, -523,
 -521, -520, -518, -517, -515, -513, -512, -510,
 -509, -507, -506, -504, -503, -501, -499, -498,
 -496, -495, -493, -492, -490, -489, -487, -485,
 -484, -482, -481, -479, -478, -476, -474, -473,
 -471, -470, -468, -467, -465, -464, -462, -460,
 -459, -457, -456, -454, -453, -451, -449, -448,
 -446, -445, -443, -442, -440, -438, -437, -435,
 -434, -432, -431, -429, -427, -426, -424, -423,
 -421, -420, -418, -416, -415, -413, -412, -410,
 -409, -407, -405, -404, -402, -401, -399, -397,
 -396, -394, -393, -391, -390, -388, -386, -385,
 -383, -382, -380, -379, -377, -375, -374, -372,
 -371, -369, -367, -366, -364, -363, -361, -360,
 -358, -356, -355, -353, -352, -350, -348, -347,
 -345, -344, -342, -341, -339, -337, -336, -334,
 -333, -331, -329, -328, -326, -325, -323, -321,
 -320, -318, -317, -315, -314, -312, -310, -309,
 -307, -306, -304, -302, -301, -299, -298, -296,
 -294, -293, -291, -290, -288, -286, -285, -283,
 -282, -280, -279, -277, -275, -274, -272, -271,
 -269, -267, -266, -264, -263, -261, -259, -258,
 -256, -255, -253, -251, -250, -248, -247, -245,
 -243, -242, -240, -239, -237, -235, -234, -232,
 -231, -229, -227, -226, -224, -223, -221, -219,
 -218, -216, -215, -213, -211, -210, -208, -207,
 -205, -203, -202, -200, -199, -197, -195, -194,
 -192, -191, -189, -187, -186, -184, -183, -181,
 -179, -178, -176, -175, -173, -171, -170, -168,
 -167, -165, -163, -162, -160, -159, -157, -155,
 -154, -152, -151, -149, -147, -146, -144, -143,
 -141, -139, -138, -136, -135, -133, -131, -130,
 -128, -126, -125, -123, -122, -120, -118, -117,
 -115, -114, -112, -110, -109, -107, -106, -104,
 -102, -101, -99, -98, -96, -94, -93, -91,
 -90, -88, -86, -85, -83, -82, -80, -78,
 -77, -75, -73, -72, -70, -69, -67, -65,
 -64, -62, -61, -59, -57, -56, -54, -53,
 -51, -49, -48, -46, -45, -43, -41, -40,
 -38, -36, -35, -33, -32, -30, -28, -27,
 -25, -24, -22, -20, -19, -17, -16, -14,
 -12, -11, -9, -8, -6, -4, -3, -1
};

// Code that generates the sine wave table above.
#if 0

#include <stdio.h>
#include <math.h>

int main()
{
    int i;
    short sine[8000];
    for ( i = 0; i < 8000; ++i ) {
        sine[i] = (short)( ((double)2048) * sin( 2 * M_PI / 8000.0 * i ) );
    }
    for ( i = 0; i < 8000; ++i ) {
        printf( " %d,", sine[i] );
        if ( ( i % 8 ) == 7 )
            printf( "\n" );
    }
    printf( "\n" );
    return 0;
}

#endif
