/****************************************************************************
**
** This file is part of the Qt Extended Opensource Package.
**
** Copyright (C) 2009 Trolltech ASA.
**
** Contact: Qt Extended Information (info@qtextended.org)
**
** This file may be used under the terms of the GNU General Public License
** version 2.0 as published by the Free Software Foundation and appearing
** in the file LICENSE.GPL included in the packaging of this file.
**
** Please review the following information to ensure GNU General Public
** Licensing requirements will be met:
**     http://www.fsf.org/licensing/licenses/info/GPLv2.html.
**
**
****************************************************************************/

#include "tst_qsimtoolkit.h"

#ifndef SYSTEMTEST

// Test encoding and decoding of EVENT DOWNLOAD envelopes based on the
// Test encoding and decoding of SEND SS commands based on the
// GCF test strings in GSM 51.010, section 27.22.4.11 - SEND SS.
void tst_QSimToolkit::testEncodeSendSS_data()
{
    QSimToolkitData::populateDataSendSS();
}
void tst_QSimToolkit::testEncodeSendSS()
{
    QFETCH( QByteArray, data );
    QFETCH( QByteArray, resp );
    QFETCH( QByteArray, returnData );
    QFETCH( int, resptype );
    QFETCH( QString, text );
    QFETCH( QString, number );
    QFETCH( int, iconId );
    QFETCH( bool, iconSelfExplanatory );
    QFETCH( QByteArray, textAttribute );
    QFETCH( QString, html );
    QFETCH( int, options );

    // Output a dummy line to give some indication of which test we are currently running.
    qDebug() << "";

    // Check that the command PDU can be parsed correctly.
    QSimCommand decoded = QSimCommand::fromPdu(data);
    QVERIFY( decoded.type() == QSimCommand::SendSS );
    QVERIFY( decoded.destinationDevice() == QSimCommand::Network );
    QCOMPARE( decoded.text(), text );
    if ( text.isEmpty() ) {
        if ( ( options & QSimCommand::EncodeEmptyStrings ) != 0 )
            QVERIFY( decoded.suppressUserFeedback() );
        else
            QVERIFY( !decoded.suppressUserFeedback() );
    } else {
        QVERIFY( !decoded.suppressUserFeedback() );
    }
    QCOMPARE( decoded.number(), number );
    QCOMPARE( (int)decoded.iconId(), iconId );
    QCOMPARE( decoded.iconSelfExplanatory(), iconSelfExplanatory );
    QCOMPARE( decoded.textAttribute(), textAttribute );
    if ( !textAttribute.isEmpty() )
        QCOMPARE( decoded.textHtml(), html );

    // Check that the original command PDU can be reconstructed correctly.
    QByteArray encoded = decoded.toPdu( (QSimCommand::ToPduOptions)options );
    QCOMPARE( encoded, data );

    // Check that the terminal response PDU can be parsed correctly.
    QSimTerminalResponse decodedResp = QSimTerminalResponse::fromPdu(resp);
    QVERIFY( data.contains( decodedResp.commandPdu() ) );
    QVERIFY( decodedResp.result() == (QSimTerminalResponse::Result)resptype );
    QCOMPARE( decodedResp.causeData(), returnData );

    // Check that the original terminal response PDU can be reconstructed correctly.
    QCOMPARE( decodedResp.toPdu(), resp );
}

// Test that SEND SS commands can be properly delivered to a client
// application and that the client application can respond appropriately.
void tst_QSimToolkit::testDeliverSendSS_data()
{
    QSimToolkitData::populateDataSendSS();
}
void tst_QSimToolkit::testDeliverSendSS()
{
    QFETCH( QByteArray, data );
    QFETCH( QByteArray, resp );
    QFETCH( QByteArray, returnData );
    QFETCH( int, resptype );
    QFETCH( QString, text );
    QFETCH( QString, number );
    QFETCH( int, iconId );
    QFETCH( bool, iconSelfExplanatory );
    QFETCH( QByteArray, textAttribute );
    QFETCH( QString, html );
    QFETCH( int, options );

    Q_UNUSED(resp);
    Q_UNUSED(returnData);
    Q_UNUSED(resptype);
    Q_UNUSED(html);

    // Output a dummy line to give some indication of which test we are currently running.
    qDebug() << "";

    // Clear the client/server state.
    server->clear();
    deliveredCommand = QSimCommand();

    // Compose and send the command.
    QSimCommand cmd;
    cmd.setType( QSimCommand::SendSS );
    cmd.setDestinationDevice( QSimCommand::Network );
    cmd.setText( text );
    if ( text.isEmpty() && ( options & QSimCommand::EncodeEmptyStrings ) != 0 )
        cmd.setSuppressUserFeedback( true );
    cmd.setNumber( number );
    cmd.setIconId( (uint)iconId );
    cmd.setIconSelfExplanatory( iconSelfExplanatory );
    cmd.setTextAttribute( textAttribute );
    server->emitCommand( cmd );

    // Wait for the command to arrive in the client.
    QVERIFY( QFutureSignal::wait( this, SIGNAL(commandSeen()), 100 ) );

    // Verify that the command was delivered exactly as we asked.
    QVERIFY( deliveredCommand.type() == cmd.type() );
    QVERIFY( deliveredCommand.text() == cmd.text() );
    QVERIFY( deliveredCommand.suppressUserFeedback() == cmd.suppressUserFeedback() );
    QVERIFY( deliveredCommand.number() == cmd.number() );
    QVERIFY( deliveredCommand.iconId() == cmd.iconId() );
    QVERIFY( deliveredCommand.iconSelfExplanatory() == cmd.iconSelfExplanatory() );
    QVERIFY( deliveredCommand.textAttribute() == cmd.textAttribute() );
    QCOMPARE( deliveredCommand.toPdu( (QSimCommand::ToPduOptions)options ), data );

    // The terminal response should have been sent immediately to ack reception of the command.
    // We cannot check the return data explicitly because that will be handled in the modem
    // and will typically be invisible to Qtopia.  We therefore compare against what the
    // response would be without the return data.
    QCOMPARE( server->responseCount(), 1 );
    QCOMPARE( server->envelopeCount(), 0 );
    QSimTerminalResponse resp2;
    resp2.setCommand( deliveredCommand );
    resp2.setResult( QSimTerminalResponse::Success );
    QCOMPARE( server->lastResponse(), resp2.toPdu() );
}

// Test the user interface in "simapp" for SEND SS.
void tst_QSimToolkit::testUISendSS_data()
{
    QSimToolkitData::populateDataSendSS();
}
void tst_QSimToolkit::testUISendSS()
{
    QFETCH( QByteArray, data );
    QFETCH( QByteArray, resp );
    QFETCH( QByteArray, returnData );
    QFETCH( int, resptype );
    QFETCH( QString, text );
    QFETCH( QString, number );
    QFETCH( int, iconId );
    QFETCH( bool, iconSelfExplanatory );
    QFETCH( QByteArray, textAttribute );
    QFETCH( QString, html );
    QFETCH( int, options );

    Q_UNUSED(html);
    Q_UNUSED(options);

    // Skip tests that we cannot test using the "simapp" UI.
    if ( resptype == 0x0004 ||      // Icon not displayed
         resptype == 0x0034 ||      // SS Return error
         resptype == 0x0032 ) {     // Command data not understood.
        QSKIP( "", SkipSingle );
    }

    // Output a dummy line to give some indication of which test we are currently running.
    qDebug() << "";

    // Create the command to be tested.
    QSimCommand cmd;
    cmd.setType( QSimCommand::SendSS );
    cmd.setDestinationDevice( QSimCommand::Network );
    cmd.setText( text );
    if ( text.isEmpty() && ( options & QSimCommand::EncodeEmptyStrings ) != 0 )
        cmd.setSuppressUserFeedback( true );
    cmd.setNumber( number );
    cmd.setIconId( (uint)iconId );
    cmd.setIconSelfExplanatory( iconSelfExplanatory );
    cmd.setTextAttribute( textAttribute );

    // Set up the server with the command, ready to be selected
    // from the "Run Test" menu item on the test menu.
    server->startUsingTestMenu( cmd );
    QVERIFY( waitForView( SimMenu::staticMetaObject ) );

    // Clear the server state just before we request the actual command under test.
    server->clear();

    // Select the first menu item.
    select();

    // Wait for the text to display.  If user feedback is suppressed, then
    // the command is supposed to be performed silently.
    if ( cmd.suppressUserFeedback() )
        QVERIFY( !waitForView( SimText::staticMetaObject ) );
    else
        QVERIFY( waitForView( SimText::staticMetaObject ) );

    // The terminal response should have been sent immediately to ack reception of the command.
    // We cannot check the return data explicitly because that will be handled in the modem
    // and will typically be invisible to Qtopia.  We therefore compare against what the
    // response would be without the return data.
    QCOMPARE( server->responseCount(), 1 );
    QCOMPARE( server->envelopeCount(), 0 );
    QSimTerminalResponse resp2;
    resp2.setCommand( deliveredCommand );
    resp2.setResult( QSimTerminalResponse::Success );
    QCOMPARE( server->lastResponse(), resp2.toPdu() );
}

#endif // !SYSTEMTEST

// Populate data-driven tests for SEND SS from the GCF test cases
// in GSM 51.010, section 27.22.4.11.
void QSimToolkitData::populateDataSendSS()
{
    QTest::addColumn<QByteArray>("data");
    QTest::addColumn<QByteArray>("resp");
    QTest::addColumn<QByteArray>("returnData");
    QTest::addColumn<int>("resptype");
    QTest::addColumn<QString>("text");
    QTest::addColumn<QString>("number");
    QTest::addColumn<int>("iconId");
    QTest::addColumn<bool>("iconSelfExplanatory");
    QTest::addColumn<QByteArray>("textAttribute");
    QTest::addColumn<QString>("html");
    QTest::addColumn<int>("options");

    // Note: we only test the SEND SS commands and TERMINAL RESPONSE's.
    // We don't test the data that is sent on the network, as we assume
    // that the modem or the modem vendor plugin will handle that automatically.

    static unsigned char const data_1_1_1a[] =
        {0xD0, 0x29, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0C, 0x43, 0x61, 0x6C, 0x6C, 0x20, 0x46, 0x6F, 0x72, 0x77, 0x61, 0x72,
         0x64, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09,
         0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB};
    static unsigned char const resp_1_1_1a[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x00,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_1_1_1a[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    QTest::newRow( "SEND SS 1.1.1A - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_1_1a, sizeof(data_1_1_1a) )
        << QByteArray( (char *)resp_1_1_1a, sizeof(resp_1_1_1a) )
        << QByteArray( (char *)retn_1_1_1a, sizeof(retn_1_1_1a) )
        << 0x0000       // Command performed successfully
        << QString( "Call Forward" )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_1_1b[] =
        {0xD0, 0x29, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0C, 0x43, 0x61, 0x6C, 0x6C, 0x20, 0x46, 0x6F, 0x72, 0x77, 0x61, 0x72,
         0x64, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09,
         0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB};
    static unsigned char const resp_1_1_1b[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x00,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_1_1_1b[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 1.1.1B - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_1_1b, sizeof(data_1_1_1b) )
        << QByteArray( (char *)resp_1_1_1b, sizeof(resp_1_1_1b) )
        << QByteArray( (char *)retn_1_1_1b, sizeof(retn_1_1_1b) )
        << 0x0000       // Command performed successfully
        << QString( "Call Forward" )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_2_1[] =
        {0xD0, 0x29, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0C, 0x43, 0x61, 0x6C, 0x6C, 0x20, 0x46, 0x6F, 0x72, 0x77, 0x61, 0x72,
         0x64, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09,
         0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB};
    static unsigned char const resp_1_2_1[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x02, 0x34,
         0x15};
    static unsigned char const retn_1_2_1[] =
        {0x15};
    QTest::newRow( "SEND SS 1.2.1 - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_2_1, sizeof(data_1_2_1) )
        << QByteArray( (char *)resp_1_2_1, sizeof(resp_1_2_1) )
        << QByteArray( (char *)retn_1_2_1, sizeof(retn_1_2_1) )
        << 0x0034       // SS Return Error
        << QString( "Call Forward" )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_3_1[] =
        {0xD0, 0x29, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0C, 0x43, 0x61, 0x6C, 0x6C, 0x20, 0x46, 0x6F, 0x72, 0x77, 0x61, 0x72,
         0x64, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09,
         0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB};
    static unsigned char const resp_1_3_1[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x02, 0x34,
         0x00};
    static unsigned char const retn_1_3_1[] =
        {0x00};
    QTest::newRow( "SEND SS 1.3.1 - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_3_1, sizeof(data_1_3_1) )
        << QByteArray( (char *)resp_1_3_1, sizeof(resp_1_3_1) )
        << QByteArray( (char *)retn_1_3_1, sizeof(retn_1_3_1) )
        << 0x0034       // SS Return Error
        << QString( "Call Forward" )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_4_1a[] =
        {0xD0, 0x2D, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0C, 0x43, 0x61, 0x6C, 0x6C, 0x20, 0x46, 0x6F, 0x72, 0x77, 0x61, 0x72,
         0x64, 0x89, 0x14, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09,
         0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43, 0x65, 0xA7, 0x11, 0xFB};
    static unsigned char const resp_1_4_1a[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x22, 0x00,
         0x0A, 0xA0, 0x1E, 0x04, 0x01, 0x21, 0x30, 0x19, 0x30, 0x17, 0x83, 0x01,
         0x10, 0x84, 0x01, 0x07, 0x89, 0x0F, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98, 0x10, 0x32, 0x54, 0x76};
    static unsigned char const retn_1_4_1a[] =
        {0x0A, 0xA0, 0x1E, 0x04, 0x01, 0x21, 0x30, 0x19, 0x30, 0x17, 0x83, 0x01,
         0x10, 0x84, 0x01, 0x07, 0x89, 0x0F, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98, 0x10, 0x32, 0x54, 0x76};
    QTest::newRow( "SEND SS 1.4.1A - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_4_1a, sizeof(data_1_4_1a) )
        << QByteArray( (char *)resp_1_4_1a, sizeof(resp_1_4_1a) )
        << QByteArray( (char *)retn_1_4_1a, sizeof(retn_1_4_1a) )
        << 0x0000       // Command performed successfully
        << QString( "Call Forward" )
        << QString( "+**21*0123456789012345678901234567*11#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_4_1b[] =
        {0xD0, 0x2D, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0C, 0x43, 0x61, 0x6C, 0x6C, 0x20, 0x46, 0x6F, 0x72, 0x77, 0x61, 0x72,
         0x64, 0x89, 0x14, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09,
         0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43, 0x65, 0xA7, 0x11, 0xFB};
    static unsigned char const resp_1_4_1b[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x00,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x11, 0x84, 0x01, 0x07};
    static unsigned char const retn_1_4_1b[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x11, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 1.4.1B - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_4_1b, sizeof(data_1_4_1b) )
        << QByteArray( (char *)resp_1_4_1b, sizeof(resp_1_4_1b) )
        << QByteArray( (char *)retn_1_4_1b, sizeof(retn_1_4_1b) )
        << 0x0000       // Command performed successfully
        << QString( "Call Forward" )
        << QString( "+**21*0123456789012345678901234567*11#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_5_1[] =
        {0xD0, 0x81, 0xFD, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83,
         0x85, 0x81, 0xEB, 0x45, 0x76, 0x65, 0x6E, 0x20, 0x69, 0x66, 0x20, 0x74,
         0x68, 0x65, 0x20, 0x46, 0x69, 0x78, 0x65, 0x64, 0x20, 0x44, 0x69, 0x61,
         0x6C, 0x6C, 0x69, 0x6E, 0x67, 0x20, 0x4E, 0x75, 0x6D, 0x62, 0x65, 0x72,
         0x20, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x20, 0x69, 0x73, 0x20,
         0x65, 0x6E, 0x61, 0x62, 0x6C, 0x65, 0x64, 0x2C, 0x20, 0x74, 0x68, 0x65,
         0x20, 0x73, 0x75, 0x70, 0x70, 0x6C, 0x65, 0x6D, 0x65, 0x6E, 0x74, 0x61,
         0x72, 0x79, 0x20, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x20, 0x63,
         0x6F, 0x6E, 0x74, 0x72, 0x6F, 0x6C, 0x20, 0x73, 0x74, 0x72, 0x69, 0x6E,
         0x67, 0x20, 0x69, 0x6E, 0x63, 0x6C, 0x75, 0x64, 0x65, 0x64, 0x20, 0x69,
         0x6E, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x45, 0x4E, 0x44, 0x20, 0x53,
         0x53, 0x20, 0x70, 0x72, 0x6F, 0x61, 0x63, 0x74, 0x69, 0x76, 0x65, 0x20,
         0x63, 0x6F, 0x6D, 0x6D, 0x61, 0x6E, 0x64, 0x20, 0x73, 0x68, 0x61, 0x6C,
         0x6C, 0x20, 0x6E, 0x6F, 0x74, 0x20, 0x62, 0x65, 0x20, 0x63, 0x68, 0x65,
         0x63, 0x6B, 0x65, 0x64, 0x20, 0x61, 0x67, 0x61, 0x69, 0x6E, 0x73, 0x74,
         0x20, 0x74, 0x68, 0x6F, 0x73, 0x65, 0x20, 0x6F, 0x66, 0x20, 0x74, 0x68,
         0x65, 0x20, 0x46, 0x44, 0x4E, 0x20, 0x6C, 0x69, 0x73, 0x74, 0x2E, 0x20,
         0x55, 0x70, 0x6F, 0x6E, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76, 0x69,
         0x6E, 0x67, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x63, 0x6F, 0x6D, 0x6D,
         0x61, 0x6E, 0x64, 0x2C, 0x20, 0x74, 0x68, 0x65, 0x20, 0x4D, 0x45, 0x20,
         0x73, 0x68, 0x61, 0x6C, 0x6C, 0x20, 0x64, 0x65, 0x63, 0x69, 0x89, 0x04,
         0xFF, 0xBA, 0x13, 0xFB};
    static unsigned char const resp_1_5_1[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x0A, 0x00,
         0x0E, 0xA4, 0x06, 0x04, 0x01, 0x06, 0x0A, 0x01, 0x02};
    static unsigned char const retn_1_5_1[] =
        {0x0E, 0xA4, 0x06, 0x04, 0x01, 0x06, 0x0A, 0x01, 0x02};
    QTest::newRow( "SEND SS 1.5.1 - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_5_1, sizeof(data_1_5_1) )
        << QByteArray( (char *)resp_1_5_1, sizeof(resp_1_5_1) )
        << QByteArray( (char *)retn_1_5_1, sizeof(retn_1_5_1) )
        << 0x0000       // Command performed successfully
        << QString( "Even if the Fixed Dialling Number service is enabled, the "
                    "supplementary service control string included in the SEND SS "
                    "proactive command shall not be checked against those of the FDN list. "
                    "Upon receiving this command, the ME shall deci" )
        << QString( "*#31#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_6_1[] =
        {0xD0, 0x1D, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x00, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09,
         0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB};
    static unsigned char const resp_1_6_1[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x00,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_1_6_1[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 1.6.1 - GCF 27.22.4.11.1" )
        << QByteArray( (char *)data_1_6_1, sizeof(data_1_6_1) )
        << QByteArray( (char *)resp_1_6_1, sizeof(resp_1_6_1) )
        << QByteArray( (char *)retn_1_6_1, sizeof(retn_1_6_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_1_1aa[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x01};
    static unsigned char const resp_2_1_1aa[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x00,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_2_1_1aa[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    QTest::newRow( "SEND SS 2.1.1AA - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_1_1aa, sizeof(data_2_1_1aa) )
        << QByteArray( (char *)resp_2_1_1aa, sizeof(resp_2_1_1aa) )
        << QByteArray( (char *)retn_2_1_1aa, sizeof(retn_2_1_1aa) )
        << 0x0000       // Command performed successfully
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_1_1ab[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x01};
    static unsigned char const resp_2_1_1ab[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x00,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_2_1_1ab[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 2.1.1AB - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_1_1ab, sizeof(data_2_1_1ab) )
        << QByteArray( (char *)resp_2_1_1ab, sizeof(resp_2_1_1ab) )
        << QByteArray( (char *)retn_2_1_1ab, sizeof(retn_2_1_1ab) )
        << 0x0000       // Command performed successfully
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_1_1ba[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x01};
    static unsigned char const resp_2_1_1ba[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x04,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_2_1_1ba[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    QTest::newRow( "SEND SS 2.1.1BA - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_1_1ba, sizeof(data_2_1_1ba) )
        << QByteArray( (char *)resp_2_1_1ba, sizeof(resp_2_1_1ba) )
        << QByteArray( (char *)retn_2_1_1ba, sizeof(retn_2_1_1ba) )
        << 0x0004       // Icon not displayed
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_1_1bb[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x01};
    static unsigned char const resp_2_1_1bb[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x04,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_2_1_1bb[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 2.1.1BB - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_1_1bb, sizeof(data_2_1_1bb) )
        << QByteArray( (char *)resp_2_1_1bb, sizeof(resp_2_1_1bb) )
        << QByteArray( (char *)retn_2_1_1bb, sizeof(retn_2_1_1bb) )
        << 0x0004       // Icon not displayed
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_2_1aa[] =
        {0xD0, 0x2C, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0B, 0x43, 0x6F, 0x6C, 0x6F, 0x75, 0x72, 0x20, 0x49, 0x63, 0x6F, 0x6E,
         0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21,
         0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x02};
    static unsigned char const resp_2_2_1aa[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x00,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_2_2_1aa[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    QTest::newRow( "SEND SS 2.2.1AA - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_2_1aa, sizeof(data_2_2_1aa) )
        << QByteArray( (char *)resp_2_2_1aa, sizeof(resp_2_2_1aa) )
        << QByteArray( (char *)retn_2_2_1aa, sizeof(retn_2_2_1aa) )
        << 0x0000       // Command performed successfully
        << QString( "Colour Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 2 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_2_1ab[] =
        {0xD0, 0x2C, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0B, 0x43, 0x6F, 0x6C, 0x6F, 0x75, 0x72, 0x20, 0x49, 0x63, 0x6F, 0x6E,
         0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21,
         0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x02};
    static unsigned char const resp_2_2_1ab[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x00,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_2_2_1ab[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 2.2.1AB - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_2_1ab, sizeof(data_2_2_1ab) )
        << QByteArray( (char *)resp_2_2_1ab, sizeof(resp_2_2_1ab) )
        << QByteArray( (char *)retn_2_2_1ab, sizeof(retn_2_2_1ab) )
        << 0x0000       // Command performed successfully
        << QString( "Colour Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 2 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_2_1ba[] =
        {0xD0, 0x2C, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0B, 0x43, 0x6F, 0x6C, 0x6F, 0x75, 0x72, 0x20, 0x49, 0x63, 0x6F, 0x6E,
         0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21,
         0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x02};
    static unsigned char const resp_2_2_1ba[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x04,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_2_2_1ba[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    QTest::newRow( "SEND SS 2.2.1BA - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_2_1ba, sizeof(data_2_2_1ba) )
        << QByteArray( (char *)resp_2_2_1ba, sizeof(resp_2_2_1ba) )
        << QByteArray( (char *)retn_2_2_1ba, sizeof(retn_2_2_1ba) )
        << 0x0004       // Icon not displayed
        << QString( "Colour Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 2 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_2_1bb[] =
        {0xD0, 0x2C, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0B, 0x43, 0x6F, 0x6C, 0x6F, 0x75, 0x72, 0x20, 0x49, 0x63, 0x6F, 0x6E,
         0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21,
         0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x00, 0x02};
    static unsigned char const resp_2_2_1bb[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x04,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_2_2_1bb[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 2.2.1BB - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_2_1bb, sizeof(data_2_2_1bb) )
        << QByteArray( (char *)resp_2_2_1bb, sizeof(resp_2_2_1bb) )
        << QByteArray( (char *)retn_2_2_1bb, sizeof(retn_2_2_1bb) )
        << 0x0004       // Icon not displayed
        << QString( "Colour Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 2 << true    // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_3_1aa[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x01, 0x01};
    static unsigned char const resp_2_3_1aa[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x00,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_2_3_1aa[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    QTest::newRow( "SEND SS 2.3.1AA - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_3_1aa, sizeof(data_2_3_1aa) )
        << QByteArray( (char *)resp_2_3_1aa, sizeof(resp_2_3_1aa) )
        << QByteArray( (char *)retn_2_3_1aa, sizeof(retn_2_3_1aa) )
        << 0x0000       // Command performed successfully
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_3_1ab[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x01, 0x01};
    static unsigned char const resp_2_3_1ab[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x00,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_2_3_1ab[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 2.3.1AB - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_3_1ab, sizeof(data_2_3_1ab) )
        << QByteArray( (char *)resp_2_3_1ab, sizeof(resp_2_3_1ab) )
        << QByteArray( (char *)retn_2_3_1ab, sizeof(retn_2_3_1ab) )
        << 0x0000       // Command performed successfully
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_3_1ba[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x01, 0x01};
    static unsigned char const resp_2_3_1ba[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x04,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_2_3_1ba[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    QTest::newRow( "SEND SS 2.3.1BA - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_3_1ba, sizeof(data_2_3_1ba) )
        << QByteArray( (char *)resp_2_3_1ba, sizeof(resp_2_3_1ba) )
        << QByteArray( (char *)retn_2_3_1ba, sizeof(retn_2_3_1ba) )
        << 0x0004       // Icon not displayed
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_3_1bb[] =
        {0xD0, 0x2B, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x0A, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x49, 0x63, 0x6F, 0x6E, 0x89,
         0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xA9, 0x01, 0xFB, 0x9E, 0x02, 0x01, 0x01};
    static unsigned char const resp_2_3_1bb[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x04,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_2_3_1bb[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    QTest::newRow( "SEND SS 2.3.1BB - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_3_1bb, sizeof(data_2_3_1bb) )
        << QByteArray( (char *)resp_2_3_1bb, sizeof(resp_2_3_1bb) )
        << QByteArray( (char *)retn_2_3_1bb, sizeof(retn_2_3_1bb) )
        << 0x0004       // Icon not displayed
        << QString( "Basic Icon" )
        << QString( "+**21*01234567890123456789*10#" )
        << 1 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_4_1[] =
        {0xD0, 0x1D, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x89,
         0x0E, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87, 0x09, 0x21, 0x43,
         0x65, 0x87, 0xB9, 0x9E, 0x02, 0x01, 0x01};
    static unsigned char const resp_2_4_1[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x32};
    QTest::newRow( "SEND SS 2.4.1 - GCF 27.22.4.11.2" )
        << QByteArray( (char *)data_2_4_1, sizeof(data_2_4_1) )
        << QByteArray( (char *)resp_2_4_1, sizeof(resp_2_4_1) )
        << QByteArray()
        << 0x0032       // Command data not understood by ME
        << QString( "" )
        << QString( "+**21*01234567890123456789#" )
        << 1 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_3_1_1a[] =
        {0xD0, 0x36, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x19, 0x80, 0x04, 0x17, 0x04, 0x14, 0x04, 0x20, 0x04, 0x10, 0x04, 0x12,
         0x04, 0x21, 0x04, 0x22, 0x04, 0x12, 0x04, 0x23, 0x04, 0x19, 0x04, 0x22,
         0x04, 0x15, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87,
         0x09, 0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB};
    static unsigned char const resp_3_1_1a[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x00,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_3_1_1a[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x89, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static ushort const str_3_1_1a[] =
        {0x0417, 0x0414, 0x0420, 0x0410, 0x0412, 0x0421, 0x0422, 0x0412, 0x0423,
         0x0419, 0x0422, 0x0415};
    QTest::newRow( "SEND SS 3.1.1A - GCF 27.22.4.11.3" )
        << QByteArray( (char *)data_3_1_1a, sizeof(data_3_1_1a) )
        << QByteArray( (char *)resp_3_1_1a, sizeof(resp_3_1_1a) )
        << QByteArray( (char *)retn_3_1_1a, sizeof(retn_3_1_1a) )
        << 0x0000       // Command performed successfully
        << QString::fromUtf16( str_3_1_1a, sizeof(str_3_1_1a) / sizeof(ushort) )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_3_1_1b[] =
        {0xD0, 0x36, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x19, 0x80, 0x04, 0x17, 0x04, 0x14, 0x04, 0x20, 0x04, 0x10, 0x04, 0x12,
         0x04, 0x21, 0x04, 0x22, 0x04, 0x12, 0x04, 0x23, 0x04, 0x19, 0x04, 0x22,
         0x04, 0x15, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21, 0x43, 0x65, 0x87,
         0x09, 0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB};
    static unsigned char const resp_3_1_1b[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x11, 0x00,
         0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static unsigned char const retn_3_1_1b[] =
        {0x0A, 0xA0, 0x0D, 0x04, 0x01, 0x21, 0x30, 0x08, 0x30, 0x06, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07};
    static ushort const str_3_1_1b[] =
        {0x0417, 0x0414, 0x0420, 0x0410, 0x0412, 0x0421, 0x0422, 0x0412, 0x0423,
         0x0419, 0x0422, 0x0415};
    QTest::newRow( "SEND SS 3.1.1B - GCF 27.22.4.11.3" )
        << QByteArray( (char *)data_3_1_1b, sizeof(data_3_1_1b) )
        << QByteArray( (char *)resp_3_1_1b, sizeof(resp_3_1_1b) )
        << QByteArray( (char *)retn_3_1_1b, sizeof(retn_3_1_1b) )
        << 0x0000       // Command performed successfully
        << QString::fromUtf16( str_3_1_1b, sizeof(str_3_1_1b) / sizeof(ushort) )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray() << QString() // No text attribute information
        << (int)( QSimCommand::NoPduOptions );

    // Test only one of the text attribute test cases.  We assume that if
    // one works, then they will all work.  The DISPLAY TEXT command tests
    // the formatting rules.
    static unsigned char const data_4_1_1[] =
        {0xD0, 0x33, 0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x81, 0x83, 0x85,
         0x10, 0x54, 0x65, 0x78, 0x74, 0x20, 0x41, 0x74, 0x74, 0x72, 0x69, 0x62,
         0x75, 0x74, 0x65, 0x20, 0x31, 0x89, 0x10, 0x91, 0xAA, 0x12, 0x0A, 0x21,
         0x43, 0x65, 0x87, 0x09, 0x21, 0x43, 0x65, 0x87, 0xA9, 0x01, 0xFB, 0xD0,
         0x04, 0x00, 0x10, 0x00, 0xB4};
    static unsigned char const resp_4_1_1[] =
        {0x81, 0x03, 0x01, 0x11, 0x00, 0x82, 0x02, 0x82, 0x81, 0x03, 0x1E, 0x00,
         0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x85, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const retn_4_1_1[] =
        {0x0A, 0xA0, 0x1A, 0x04, 0x01, 0x21, 0x30, 0x15, 0x30, 0x13, 0x83, 0x01,
         0x00, 0x84, 0x01, 0x07, 0x85, 0x0B, 0x91, 0x10, 0x32, 0x54, 0x76, 0x98,
         0x10, 0x32, 0x54, 0x76, 0x98};
    static unsigned char const attr_4_1_1[] =
        {0x00, 0x10, 0x00, 0xB4};
    QTest::newRow( "SEND SS 4.1.1 - GCF 27.22.4.11.4" )
        << QByteArray( (char *)data_4_1_1, sizeof(data_4_1_1) )
        << QByteArray( (char *)resp_4_1_1, sizeof(resp_4_1_1) )
        << QByteArray( (char *)retn_4_1_1, sizeof(retn_4_1_1) )
        << 0x0000       // Command performed successfully
        << QString( "Text Attribute 1" )
        << QString( "+**21*01234567890123456789*10#" )
        << 0 << false   // Icon details
        << QByteArray( (char *)attr_4_1_1, sizeof(attr_4_1_1) )
        << QString( "<body bgcolor=\"#FFFF00\"><div align=\"left\"><font color=\"#008000\">Text Attribute 1</font></div></body>" )
        << (int)( QSimCommand::NoPduOptions );
}
