/****************************************************************************
**
** This file is part of the Qt Extended Opensource Package.
**
** Copyright (C) 2009 Trolltech ASA.
**
** Contact: Qt Extended Information (info@qtextended.org)
**
** This file may be used under the terms of the GNU General Public License
** version 2.0 as published by the Free Software Foundation and appearing
** in the file LICENSE.GPL included in the packaging of this file.
**
** Please review the following information to ensure GNU General Public
** Licensing requirements will be met:
**     http://www.fsf.org/licensing/licenses/info/GPLv2.html.
**
**
****************************************************************************/

#include "tst_qsimtoolkit.h"

#include <QPhoneCallManager>

#ifndef SYSTEMTEST

// Test encoding and decoding of EVENT DOWNLOAD envelopes based on the
// Test encoding and decoding of OPEN CHANNEL commands based on the
// GCF test strings in GSM 51.010, section 27.22.4.27 - OPEN CHANNEL.
void tst_QSimToolkit::testEncodeOpenChannel_data()
{
    QSimToolkitData::populateDataOpenChannel();
}
void tst_QSimToolkit::testEncodeOpenChannel()
{
    QFETCH( QByteArray, data );
    QFETCH( QByteArray, resp );
    QFETCH( QByteArray, parameters );
    QFETCH( QByteArray, respParameters );
    QFETCH( int, resptype );
    QFETCH( QString, text );
    QFETCH( QString, number );
    QFETCH( QString, subAddress );
    QFETCH( int, linkOptions );
    QFETCH( int, iconId );
    QFETCH( bool, iconSelfExplanatory );
    QFETCH( int, options );

    // Output a dummy line to give some indication of which test we are currently running.
    qDebug() << "";

    // Check that the command PDU can be parsed correctly.
    QSimCommand decoded = QSimCommand::fromPdu(data);
    QVERIFY( decoded.type() == QSimCommand::OpenChannel );
    QVERIFY( decoded.destinationDevice() == QSimCommand::ME );
    QCOMPARE( decoded.text(), text );
    QCOMPARE( decoded.number(), number );
    QCOMPARE( decoded.subAddress(), subAddress );
    QCOMPARE( decoded.qualifier(), linkOptions );
    QCOMPARE( decoded.extensionData(), parameters );
    QCOMPARE( (int)decoded.iconId(), iconId );
    QCOMPARE( decoded.iconSelfExplanatory(), iconSelfExplanatory );

    // Check that the original command PDU can be reconstructed correctly.
    QByteArray encoded = decoded.toPdu( (QSimCommand::ToPduOptions)options );
    QCOMPARE( encoded, data );

    // Check that the terminal response PDU can be parsed correctly.
    QSimTerminalResponse decodedResp = QSimTerminalResponse::fromPdu(resp);
    QVERIFY( data.contains( decodedResp.commandPdu() ) );
    if ( resptype < 0x0100 ) {
        QVERIFY( decodedResp.result() == (QSimTerminalResponse::Result)resptype );
        QVERIFY( decodedResp.causeData().isEmpty() );
        QVERIFY( decodedResp.cause() == QSimTerminalResponse::NoSpecificCause );
    } else {
        QVERIFY( decodedResp.result() == (QSimTerminalResponse::Result)(resptype >> 8) );
        QVERIFY( decodedResp.causeData().size() == 1 );
        QVERIFY( decodedResp.cause() == (QSimTerminalResponse::Cause)(resptype & 0xFF) );
    }
    QCOMPARE( decodedResp.extensionData(), respParameters );

    // Check that the original terminal response PDU can be reconstructed correctly.
    QCOMPARE( decodedResp.toPdu(), resp );
}

// Test that OPEN CHANNEL commands can be properly delivered to a client
// application and that the client application can respond appropriately.
void tst_QSimToolkit::testDeliverOpenChannel_data()
{
    QSimToolkitData::populateDataOpenChannel();
}
void tst_QSimToolkit::testDeliverOpenChannel()
{
    QFETCH( QByteArray, data );
    QFETCH( QByteArray, resp );
    QFETCH( QByteArray, parameters );
    QFETCH( QByteArray, respParameters );
    QFETCH( int, resptype );
    QFETCH( QString, text );
    QFETCH( QString, number );
    QFETCH( QString, subAddress );
    QFETCH( int, linkOptions );
    QFETCH( int, iconId );
    QFETCH( bool, iconSelfExplanatory );
    QFETCH( int, options );

    // Output a dummy line to give some indication of which test we are currently running.
    qDebug() << "";

    // Clear the client/server state.
    server->clear();
    deliveredCommand = QSimCommand();

    // Compose and send the command.
    QSimCommand cmd;
    cmd.setType( QSimCommand::OpenChannel );
    cmd.setText( text );
    cmd.setNumber( number );
    cmd.setSubAddress( subAddress );
    cmd.setQualifier( linkOptions );
    cmd.setIconId( (uint)iconId );
    cmd.setIconSelfExplanatory( iconSelfExplanatory );
    cmd.setExtensionData( parameters );
    server->emitCommand( cmd );

    // Wait for the command to arrive in the client.
    QVERIFY( QFutureSignal::wait( this, SIGNAL(commandSeen()), 100 ) );

    // Verify that the command was delivered exactly as we asked.
    QVERIFY( deliveredCommand.type() == cmd.type() );
    QVERIFY( deliveredCommand.text() == cmd.text() );
    QVERIFY( deliveredCommand.number() == cmd.number() );
    QVERIFY( deliveredCommand.subAddress() == cmd.subAddress() );
    QVERIFY( deliveredCommand.qualifier() == cmd.qualifier() );
    QVERIFY( deliveredCommand.iconId() == cmd.iconId() );
    QVERIFY( deliveredCommand.iconSelfExplanatory() == cmd.iconSelfExplanatory() );
    QVERIFY( deliveredCommand.extensionData() == cmd.extensionData() );
    QCOMPARE( deliveredCommand.toPdu( (QSimCommand::ToPduOptions)options ), data );

    // There should be no responses or envelopes in the reverse direction yet.
    QCOMPARE( server->responseCount(), 0 );
    QCOMPARE( server->envelopeCount(), 0 );

    // Compose and send the response.
    QSimTerminalResponse response;
    response.setCommand( deliveredCommand );
    if ( resptype < 0x0100 ) {
        response.setResult( (QSimTerminalResponse::Result)resptype );
    } else {
        response.setResult( (QSimTerminalResponse::Result)(resptype >> 8) );
        response.setCause( (QSimTerminalResponse::Cause)(resptype & 0xFF) );
    }
    response.setExtensionData( respParameters );
    client->sendResponse( response );

    // Wait for the response to be received.
    QVERIFY( QFutureSignal::wait( server, SIGNAL(responseSeen()), 100 ) );

    qDebug() << QTest::toHexRepresentation( resp.constData(), resp.length() );

    // Check that the response is what we expected to get.
    QCOMPARE( server->responseCount(), 1 );
    QCOMPARE( server->envelopeCount(), 0 );
    QCOMPARE( server->lastResponse(), resp );
}

// Test the user interface in "simapp" for OPEN CHANNEL.
void tst_QSimToolkit::testUIOpenChannel_data()
{
    QSimToolkitData::populateDataOpenChannel();
}

void tst_QSimToolkit::testUIOpenChannel()
{
    QFETCH( QByteArray, data );
    QFETCH( QByteArray, resp );
    QFETCH( QByteArray, parameters );
    QFETCH( QByteArray, respParameters );
    QFETCH( int, resptype );
    QFETCH( QString, text );
    QFETCH( QString, number );
    QFETCH( QString, subAddress );
    QFETCH( int, linkOptions );
    QFETCH( int, iconId );
    QFETCH( bool, iconSelfExplanatory );
    QFETCH( int, options );

    Q_UNUSED(options)

    // Skip tests that we cannot test using the "simapp" UI.
    if ( resptype == 0x2100 ||      // Network unable to process command
            resptype == 0x0007 ||   // Command performed with modification
            resptype == 0x3A01 ) {  // No channel available
        QSKIP( "", SkipSingle );
    }

    // Output a dummy line to give some indication of which test we are currently running.
    qDebug() << "";

    // Create the command to be tested.
    QSimCommand cmd;
    cmd.setType( QSimCommand::OpenChannel );
    cmd.setText( text );
    cmd.setNumber( number );
    cmd.setSubAddress( subAddress );
    cmd.setQualifier( linkOptions );
    cmd.setIconId( (uint)iconId );
    cmd.setIconSelfExplanatory( iconSelfExplanatory );
    cmd.setExtensionData( parameters );

    // Set up some dummy calls if needed before sending the command
    QPhoneCallManager callManager;
    if ( cmd.disposition() != QSimCommand::IfNoOtherCalls
            || ( cmd.disposition() == QSimCommand::IfNoOtherCalls
                && resptype == 0x2002 ) ) { // ME busy on another call
        QDialOptions dialopts;
        dialopts.setNumber( "3333" );
        QPhoneCall call = callManager.create( "Voice" );

        if ( call.isNull() )
            qDebug() << "Failed to create a dummy call";

        call.dial( dialopts );

        msleep( 3000 ); // wait for 3 seconds for the call connected.
    }

    // Set up the server with the command, ready to be selected
    // from the "Run Test" menu item on the test menu.
    server->startUsingTestMenu( cmd );
    QVERIFY( waitForView( SimMenu::staticMetaObject ) );

    // Clear the server state just before we request the actual command under test.
    server->clear();

    // Select the first menu item.
    select();

    // Wait for the text to display.
    QVERIFY( waitForView( SimChannel::staticMetaObject ) );

    // Wait for icons to load.  Should be fast because they are in-process.
    if ( iconId != 0 )
        msleep(1000);

    // Determine what needs to be done next.
    if ( resptype == 0x0022 ) { // User did not accept the request
        keyClick( Qt::Key_Back );

        QVERIFY( QFutureSignal::wait( server, SIGNAL(responseSeen()), 5000 ) );
    } else if ( resptype == 0x2002 ) { // ME busy on another call
        keyClick( Qt::Key_Back ); // click OK to information

        QVERIFY( QFutureSignal::wait( server, SIGNAL(responseSeen()), 5000 ) );
    } else { // success
        // place a call by clicking contex 1 button
        keyClick( Qt::Key_Context1 );

        QVERIFY( QFutureSignal::wait( server, SIGNAL(responseSeen()), 5000 ) );
    }

    // Check that the response is what we expected.
    //QCOMPARE( server->responseCount(), 1 );
    //QCOMPARE( server->envelopeCount(), 0 );
    //QCOMPARE( server->lastResponse(), resp );

    // We cannot check the extension data explicitly because that will be handled in the modem
    // and will typically be invisible to Qtopia.  We therefore compare against what the
    // response would be without the extension data.
    QCOMPARE( server->responseCount(), 1 );
    QCOMPARE( server->envelopeCount(), 0 );
    QSimTerminalResponse resp2;
    resp2.setCommand( deliveredCommand );
    if ( resptype == 0x2002 ) { // Busy on Call
        resp2.setResult( QSimTerminalResponse::MEUnableToProcess );
        resp2.setCause( QSimTerminalResponse::BusyOnCall );
    } else {
        resp2.setResult( (QSimTerminalResponse::Result)resptype );
    }
    QCOMPARE( server->lastResponse(), resp2.toPdu() );

    // hangup all calls
    foreach ( QPhoneCall call, callManager.calls() ) {
        call.hangup();
    }
}


#endif // !SYSTEMTEST

// Populate data-driven tests for OPEN CHANNEL from the GCF test cases
// in GSM 51.010, section 27.22.4.27.
void QSimToolkitData::populateDataOpenChannel()
{
    QTest::addColumn<QByteArray>("data");
    QTest::addColumn<QByteArray>("resp");
    QTest::addColumn<QByteArray>("parameters");
    QTest::addColumn<QByteArray>("respParameters");
    QTest::addColumn<int>("resptype");
    QTest::addColumn<QString>("text");
    QTest::addColumn<QString>("number");
    QTest::addColumn<QString>("subAddress");
    QTest::addColumn<int>("linkOptions");
    QTest::addColumn<int>("iconId");
    QTest::addColumn<bool>("iconSelfExplanatory");
    QTest::addColumn<int>("options");

    // Note: section 27.22.4.27.1 is "Void" in GSM 51.010-4.  The following
    // CSD-related test cases are from GSM 11.10-4.

    static unsigned char const data_1_1_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x07, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_1_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_1_1[] =
        {0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_1_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.1.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_1_1, sizeof(data_1_1_1) )
        << QByteArray( (char *)resp_1_1_1, sizeof(resp_1_1_1) )
        << QByteArray( (char *)parm_1_1_1, sizeof(parm_1_1_1) )
        << QByteArray( (char *)rpar_1_1_1, sizeof(rpar_1_1_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_2_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x0C, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_2_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x0C, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_2_1[] =
        {0xB5, 0x04, 0x01, 0x0C, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_2_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x0C, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.2.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_2_1, sizeof(data_1_2_1) )
        << QByteArray( (char *)resp_1_2_1, sizeof(resp_1_2_1) )
        << QByteArray( (char *)parm_1_2_1, sizeof(parm_1_2_1) )
        << QByteArray( (char *)rpar_1_2_1, sizeof(rpar_1_2_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_3_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x27, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_3_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x27, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_3_1[] =
        {0xB5, 0x04, 0x01, 0x27, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_3_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x27, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.3.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_3_1, sizeof(data_1_3_1) )
        << QByteArray( (char *)resp_1_3_1, sizeof(resp_1_3_1) )
        << QByteArray( (char *)parm_1_3_1, sizeof(parm_1_3_1) )
        << QByteArray( (char *)rpar_1_3_1, sizeof(rpar_1_3_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_4_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x47, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_4_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x47, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_4_1[] =
        {0xB5, 0x04, 0x01, 0x47, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_4_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x47, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.4.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_4_1, sizeof(data_1_4_1) )
        << QByteArray( (char *)resp_1_4_1, sizeof(resp_1_4_1) )
        << QByteArray( (char *)parm_1_4_1, sizeof(parm_1_4_1) )
        << QByteArray( (char *)rpar_1_4_1, sizeof(rpar_1_4_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_5_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x07, 0x04, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_5_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x04, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_5_1[] =
        {0xB5, 0x04, 0x01, 0x07, 0x04, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_5_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x04, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.5.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_5_1, sizeof(data_1_5_1) )
        << QByteArray( (char *)resp_1_5_1, sizeof(resp_1_5_1) )
        << QByteArray( (char *)parm_1_5_1, sizeof(parm_1_5_1) )
        << QByteArray( (char *)rpar_1_5_1, sizeof(rpar_1_5_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_6_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x07, 0x00, 0x02, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_6_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x00, 0x02, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_6_1[] =
        {0xB5, 0x04, 0x01, 0x07, 0x00, 0x02, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_6_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x00, 0x02, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.6.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_6_1, sizeof(data_1_6_1) )
        << QByteArray( (char *)resp_1_6_1, sizeof(resp_1_6_1) )
        << QByteArray( (char *)parm_1_6_1, sizeof(parm_1_6_1) )
        << QByteArray( (char *)rpar_1_6_1, sizeof(rpar_1_6_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_7_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x54, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_7_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x07,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_7_1[] =
        {0xB5, 0x04, 0x01, 0x54, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_7_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.7.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_7_1, sizeof(data_1_7_1) )
        << QByteArray( (char *)resp_1_7_1, sizeof(resp_1_7_1) )
        << QByteArray( (char *)parm_1_7_1, sizeof(parm_1_7_1) )
        << QByteArray( (char *)rpar_1_7_1, sizeof(rpar_1_7_1) )
        << 0x0007       // Command performed with modification
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_8_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x54, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_8_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x02, 0x21,
         0x00, 0xB5, 0x04, 0x01, 0x54, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const parm_1_8_1[] =
        {0xB5, 0x04, 0x01, 0x54, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_8_1[] =
        {0xB5, 0x04, 0x01, 0x54, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.8.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_8_1, sizeof(data_1_8_1) )
        << QByteArray( (char *)resp_1_8_1, sizeof(resp_1_8_1) )
        << QByteArray( (char *)parm_1_8_1, sizeof(parm_1_8_1) )
        << QByteArray( (char *)rpar_1_8_1, sizeof(rpar_1_8_1) )
        << 0x2100       // Network unable to process command
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_9_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x33, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_9_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x33, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    static unsigned char const parm_1_9_1[] =
        {0xB5, 0x04, 0x01, 0x33, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_9_1[] =
        {0xB8, 0x02, 0x81, 0x00, 0xB5, 0x04, 0x01, 0x33, 0x00, 0x01, 0xB9, 0x02,
         0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.9.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_9_1, sizeof(data_1_9_1) )
        << QByteArray( (char *)resp_1_9_1, sizeof(resp_1_9_1) )
        << QByteArray( (char *)parm_1_9_1, sizeof(parm_1_9_1) )
        << QByteArray( (char *)rpar_1_9_1, sizeof(rpar_1_9_1) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_9_2[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x33, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_9_2[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x02, 0x3A,
         0x01, 0xB5, 0x04, 0x01, 0x33, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const parm_1_9_2[] =
        {0xB5, 0x04, 0x01, 0x33, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_9_2[] =
        {0xB5, 0x04, 0x01, 0x33, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.9.2 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_9_2, sizeof(data_1_9_2) )
        << QByteArray( (char *)resp_1_9_2, sizeof(resp_1_9_2) )
        << QByteArray( (char *)parm_1_9_2, sizeof(parm_1_9_2) )
        << QByteArray( (char *)rpar_1_9_2, sizeof(rpar_1_9_2) )
        << 0x3A01       // No channel available
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_1_10_1[] =
        {0xD0, 0x1E, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x86,
         0x09, 0x91, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0xF8, 0xB5, 0x04,
         0x01, 0x07, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const resp_1_10_1[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x02, 0x20,
         0x02, 0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const parm_1_10_1[] =
        {0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    static unsigned char const rpar_1_10_1[] =
        {0xB5, 0x04, 0x01, 0x07, 0x00, 0x01, 0xB9, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 1.10.1 - GCF 27.22.4.27.1" )
        << QByteArray( (char *)data_1_10_1, sizeof(data_1_10_1) )
        << QByteArray( (char *)resp_1_10_1, sizeof(resp_1_10_1) )
        << QByteArray( (char *)parm_1_10_1, sizeof(parm_1_10_1) )
        << QByteArray( (char *)rpar_1_10_1, sizeof(rpar_1_10_1) )
        << 0x2002       // ME currently busy on call
        << QString( "" )
        << QString( "+112233445566778" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_1_1a[] =
        {0xD0, 0x36, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_1_1a[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_1_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_1_1a[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.1.1A - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_1_1a, sizeof(data_2_1_1a) )
        << QByteArray( (char *)resp_2_1_1a, sizeof(resp_2_1_1a) )
        << QByteArray( (char *)parm_2_1_1a, sizeof(parm_2_1_1a) )
        << QByteArray( (char *)rpar_2_1_1a, sizeof(rpar_2_1_1a) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_1_1b[] =
        {0xD0, 0x36, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_1_1b[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_1_1b[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_1_1b[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.1.1B - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_1_1b, sizeof(data_2_1_1b) )
        << QByteArray( (char *)resp_2_1_1b, sizeof(resp_2_1_1b) )
        << QByteArray( (char *)parm_2_1_1b, sizeof(parm_2_1_1b) )
        << QByteArray( (char *)rpar_2_1_1b, sizeof(rpar_2_1_1b) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_2_1a[] =
        {0xD0, 0x42, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78,
         0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_2_1a[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_2_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_2_1a[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.2.1A - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_2_1a, sizeof(data_2_2_1a) )
        << QByteArray( (char *)resp_2_2_1a, sizeof(resp_2_2_1a) )
        << QByteArray( (char *)parm_2_2_1a, sizeof(parm_2_2_1a) )
        << QByteArray( (char *)rpar_2_2_1a, sizeof(rpar_2_2_1a) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_2_1b[] =
        {0xD0, 0x42, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78,
         0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_2_1b[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_2_1b[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_2_1b[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.2.1B - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_2_1b, sizeof(data_2_2_1b) )
        << QByteArray( (char *)resp_2_2_1b, sizeof(resp_2_2_1b) )
        << QByteArray( (char *)parm_2_2_1b, sizeof(parm_2_2_1b) )
        << QByteArray( (char *)rpar_2_2_1b, sizeof(rpar_2_2_1b) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_3_1a[] =
        {0xD0, 0x4B, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x05,
         0x07, 0x4F, 0x70, 0x65, 0x6E, 0x20, 0x49, 0x44, 0x35, 0x07, 0x02, 0x02,
         0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78, 0x47, 0x0A, 0x06,
         0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73, 0x0D, 0x08, 0xF4,
         0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73,
         0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05,
         0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_3_1a[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_3_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_3_1a[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.3.1A - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_3_1a, sizeof(data_2_3_1a) )
        << QByteArray( (char *)resp_2_3_1a, sizeof(resp_2_3_1a) )
        << QByteArray( (char *)parm_2_3_1a, sizeof(parm_2_3_1a) )
        << QByteArray( (char *)rpar_2_3_1a, sizeof(rpar_2_3_1a) )
        << 0x0000       // Command performed successfully
        << QString( "Open ID" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_3_1b[] =
        {0xD0, 0x4B, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x05,
         0x07, 0x4F, 0x70, 0x65, 0x6E, 0x20, 0x49, 0x44, 0x35, 0x07, 0x02, 0x02,
         0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78, 0x47, 0x0A, 0x06,
         0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73, 0x0D, 0x08, 0xF4,
         0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73,
         0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05,
         0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_3_1b[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_3_1b[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_3_1b[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.3.1B - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_3_1b, sizeof(data_2_3_1b) )
        << QByteArray( (char *)resp_2_3_1b, sizeof(resp_2_3_1b) )
        << QByteArray( (char *)parm_2_3_1b, sizeof(parm_2_3_1b) )
        << QByteArray( (char *)rpar_2_3_1b, sizeof(rpar_2_3_1b) )
        << 0x0000       // Command performed successfully
        << QString( "Open ID" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_4_1a[] =
        {0xD0, 0x46, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x05,
         0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02,
         0x05, 0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02,
         0x72, 0x73, 0x3E, 0x00, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C,
         0x6F, 0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64,
         0x3C, 0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_4_1a[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_4_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x3E, 0x00, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F,
         0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C,
         0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_4_1a[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.4.1A - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_4_1a, sizeof(data_2_4_1a) )
        << QByteArray( (char *)resp_2_4_1a, sizeof(resp_2_4_1a) )
        << QByteArray( (char *)parm_2_4_1a, sizeof(parm_2_4_1a) )
        << QByteArray( (char *)rpar_2_4_1a, sizeof(rpar_2_4_1a) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_4_1b[] =
        {0xD0, 0x46, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x05,
         0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02,
         0x05, 0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02,
         0x72, 0x73, 0x3E, 0x00, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C,
         0x6F, 0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64,
         0x3C, 0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_4_1b[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x00,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_4_1b[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x3E, 0x00, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F,
         0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C,
         0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_4_1b[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.4.1B - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_4_1b, sizeof(data_2_4_1b) )
        << QByteArray( (char *)resp_2_4_1b, sizeof(resp_2_4_1b) )
        << QByteArray( (char *)parm_2_4_1b, sizeof(parm_2_4_1b) )
        << QByteArray( (char *)rpar_2_4_1b, sizeof(rpar_2_4_1b) )
        << 0x0000       // Command performed successfully
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::EncodeEmptyStrings );

    static unsigned char const data_2_5_1a[] =
        {0xD0, 0x42, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0xFF, 0xFF,
         0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_5_1a[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x07,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_5_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0xFF,
         0xFF, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_5_1a[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.5.1A - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_5_1a, sizeof(data_2_5_1a) )
        << QByteArray( (char *)resp_2_5_1a, sizeof(resp_2_5_1a) )
        << QByteArray( (char *)parm_2_5_1a, sizeof(parm_2_5_1a) )
        << QByteArray( (char *)rpar_2_5_1a, sizeof(rpar_2_5_1a) )
        << 0x0007       // Command performed with modifications
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_5_1b[] =
        {0xD0, 0x42, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0xFF, 0xFF,
         0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_5_1b[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x07,
         0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_5_1b[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0xFF,
         0xFF, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_5_1b[] =
        {0x38, 0x02, 0x81, 0x00, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F,
         0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.5.1B - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_5_1b, sizeof(data_2_5_1b) )
        << QByteArray( (char *)resp_2_5_1b, sizeof(resp_2_5_1b) )
        << QByteArray( (char *)parm_2_5_1b, sizeof(parm_2_5_1b) )
        << QByteArray( (char *)rpar_2_5_1b, sizeof(rpar_2_5_1b) )
        << 0x0007       // Command performed with modifications
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    // OPEN CHANNEL 2.6.1 is marked as "Void" in both GSM 11.10-4 and 51.010-4.

    static unsigned char const data_2_7_1a[] =
        {0xD0, 0x4B, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x05,
         0x07, 0x4F, 0x70, 0x65, 0x6E, 0x20, 0x49, 0x44, 0x35, 0x07, 0x02, 0x02,
         0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78, 0x47, 0x0A, 0x06,
         0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73, 0x0D, 0x08, 0xF4,
         0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73,
         0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05,
         0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_7_1a[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x22,
         0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_7_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_7_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.7.1A - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_7_1a, sizeof(data_2_7_1a) )
        << QByteArray( (char *)resp_2_7_1a, sizeof(resp_2_7_1a) )
        << QByteArray( (char *)parm_2_7_1a, sizeof(parm_2_7_1a) )
        << QByteArray( (char *)rpar_2_7_1a, sizeof(rpar_2_7_1a) )
        << 0x0022       // User did not accept the proactive command
        << QString( "Open ID" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_7_1b[] =
        {0xD0, 0x4B, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x05,
         0x07, 0x4F, 0x70, 0x65, 0x6E, 0x20, 0x49, 0x44, 0x35, 0x07, 0x02, 0x02,
         0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78, 0x47, 0x0A, 0x06,
         0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73, 0x0D, 0x08, 0xF4,
         0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08, 0xF4, 0x55, 0x73,
         0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD, 0x9C, 0x3E, 0x05,
         0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_7_1b[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x01, 0x22,
         0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_7_1b[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_7_1b[] =
        {0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.7.1B - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_7_1b, sizeof(data_2_7_1b) )
        << QByteArray( (char *)resp_2_7_1b, sizeof(resp_2_7_1b) )
        << QByteArray( (char *)parm_2_7_1b, sizeof(parm_2_7_1b) )
        << QByteArray( (char *)rpar_2_7_1b, sizeof(rpar_2_7_1b) )
        << 0x0022       // User did not accept the proactive command
        << QString( "Open ID" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_8_1a[] =
        {0xD0, 0x42, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78,
         0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_8_1a[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x02, 0x20,
         0x02, 0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_8_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_8_1a[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.8.1A - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_8_1a, sizeof(data_2_8_1a) )
        << QByteArray( (char *)resp_2_8_1a, sizeof(resp_2_8_1a) )
        << QByteArray( (char *)parm_2_8_1a, sizeof(parm_2_8_1a) )
        << QByteArray( (char *)rpar_2_8_1a, sizeof(rpar_2_8_1a) )
        << 0x2002       // ME busy on call
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );

    static unsigned char const data_2_8_1b[] =
        {0xD0, 0x42, 0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x81, 0x82, 0x35,
         0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78,
         0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72, 0x73,
         0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D, 0x08,
         0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01, 0xAD,
         0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const resp_2_8_1b[] =
        {0x81, 0x03, 0x01, 0x40, 0x01, 0x82, 0x02, 0x82, 0x81, 0x83, 0x02, 0x20,
         0x02, 0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    static unsigned char const parm_2_8_1b[] =
        {0x35, 0x07, 0x02, 0x02, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05,
         0x78, 0x47, 0x0A, 0x06, 0x54, 0x65, 0x73, 0x74, 0x47, 0x70, 0x02, 0x72,
         0x73, 0x0D, 0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x4C, 0x6F, 0x67, 0x0D,
         0x08, 0xF4, 0x55, 0x73, 0x65, 0x72, 0x50, 0x77, 0x64, 0x3C, 0x03, 0x01,
         0xAD, 0x9C, 0x3E, 0x05, 0x21, 0x01, 0x01, 0x01, 0x01};
    static unsigned char const rpar_2_8_1b[] =
        {0x35, 0x07, 0x02, 0x00, 0x04, 0x05, 0x05, 0x1F, 0x02, 0x39, 0x02, 0x05, 0x78};
    QTest::newRow( "OPEN CHANNEL 2.8.1B - GCF 27.22.4.27.2" )
        << QByteArray( (char *)data_2_8_1b, sizeof(data_2_8_1b) )
        << QByteArray( (char *)resp_2_8_1b, sizeof(resp_2_8_1b) )
        << QByteArray( (char *)parm_2_8_1b, sizeof(parm_2_8_1b) )
        << QByteArray( (char *)rpar_2_8_1b, sizeof(rpar_2_8_1b) )
        << 0x2002       // ME busy on call
        << QString( "" )
        << QString( "" )
        << QString( "" )
        << Open_Immediate
        << 0 << false   // No icon
        << (int)( QSimCommand::NoPduOptions );
}
