/****************************************************************************
**
** This file is part of the Qt Extended Opensource Package.
**
** Copyright (C) 2009 Trolltech ASA.
**
** Contact: Qt Extended Information (info@qtextended.org)
**
** This file may be used under the terms of the GNU General Public License
** version 2.0 as published by the Free Software Foundation and appearing
** in the file LICENSE.GPL included in the packaging of this file.
**
** Please review the following information to ensure GNU General Public
** Licensing requirements will be met:
**     http://www.fsf.org/licensing/licenses/info/GPLv2.html.
**
**
****************************************************************************/

/*!
  \class QThumbnail
    \inpublicgroup QtBaseModule

  \brief The QThumbnail class provides thumbnails of large images.

  A thumbnail is a small version of an image loaded from a file or QIODevice.

  Thumbnail pixmaps are generated by calling \c pixmap(maximum_size).
  Usually, aspect ratio is preserved, so the actualSize() of the image
  may have a smaller width or height than the maximum size.

  For many image formats, thumbnails
  are loaded by methods more efficient than the naive method of simply loading the entire image
  and scaling it down. For other formats, the naive method is used.

  This code paints the contents of \c image.jpg centered on the calling widget:

  \code
    QPainter painter( this );

    QThumbnail thumbnail( "image.jpg" );
    offset = (size() - thumbnail.actualSize(size())) / 2;
    painter.drawPixmap( offset.width(), offset.height(), thumbnail.pixmap(size()));
  \endcode

  \ingroup multimedia
*/

#include "qthumbnail.h"

#include <QByteArray>
#include <QFile>
#include <QImage>
#include <QImageReader>
#include <QImageIOHandler>
#include <QVariant>
#include <qglobal.h>
#include <QDebug>

class QThumbnailPrivate
{
public:
    QImageReader *reader;
};

#define JPEG_DECOMPRESSION_QUALITY 10

/*!
  Constructs the thumbnail for the given image \a fileName.
  The file is not loaded at this time, but it is opened.
*/
QThumbnail::QThumbnail( const QString& fileName )
{
    d = new QThumbnailPrivate;

    d->reader = new QImageReader( fileName );

    // Enable fast decompression if supported
    if ( d->reader->supportsOption( QImageIOHandler::Quality ) ) {
        //only jpeg supports this at the moment
        QString format = d->reader->format().toLower();
        if ( format == QLatin1String("jpeg") || format == QLatin1String("jpg") )
            d->reader->setQuality( JPEG_DECOMPRESSION_QUALITY );
    }
}

/*!
  Constructs the thumbnail for an image to be read from the given io \a device.
*/
QThumbnail::QThumbnail( QIODevice *device )
{
    d = new QThumbnailPrivate;

    d->reader = new QImageReader( device );

    // Enable fast decompression if supported
    if ( d->reader->supportsOption( QImageIOHandler::Quality ) ) {
        //only jpeg supports this at the moment
        QString format = d->reader->format().toLower();
        if ( format == QLatin1String("jpeg") || format == QLatin1String("jpg") )
            d->reader->setQuality( JPEG_DECOMPRESSION_QUALITY );
    }
}

/*!
  Destroys the thumbnail.
*/
QThumbnail::~QThumbnail()
{
    delete d->reader;
    delete d;
}

/*!
  Returns the actual size of the thumbnail for the requested \a size and \a
  mode. An invalid size is returned if the image is not able to be read.
  The whole image may be loaded if the image handler does not support the
  QImageIOHandler::Size option.

  If \a size is invalid the function returns the size as defined by
  QImageIOHandler::Size. This can be used to prevent that the image is scaled
  up when calling QThumbnail::pixmap():

  \code
      thumbnail.pixmap(
          thumbnail.actualSize(
              maxsize.boundedTo(
                  thumbnail.actualSize(QSize())
              )
          )
      )
  \endcode

  \sa pixmap()
*/
QSize QThumbnail::actualSize( const QSize& size, Qt::AspectRatioMode mode )
{
    if( mode == Qt::IgnoreAspectRatio ) {
        return size;
    }

    QSize actual;

    if( d->reader->supportsOption( QImageIOHandler::Size ) ) {
        actual = d->reader->size();
        if (size.isValid())
            actual.scale( size, mode );
    } else {
        // Load image to determine size
        QImage image = d->reader->read();
        actual = image.size();
        if (size.isValid())
            actual.scale( size, mode );
    }

    return actual;
}

/*!
  Returns a pixmap with the requested \a size and \a mode.  A null pixmap
  is returned if the image cannot be loaded.

  The pixmap will be generated by scaling the image as it is being loaded
  where possible. This is reliant on the support for the
  QImageIOHandler::ScaledSize option in the image handler. If the option is
  not supported the whole image will be loaded and then scaled with the
  requested \a transformationMode.

  \sa actualSize()
*/
QPixmap QThumbnail::pixmap( const QSize& size, Qt::AspectRatioMode mode, Qt::TransformationMode transformationMode )
{
    // If supported, use handler to scale image
    // Otherwise, load then scale image
    bool notScaled = true;
    if ( d->reader->supportsOption( QImageIOHandler::ScaledSize ) ) {
        d->reader->setQuality( 49 ); // Otherwise Qt smooth scales
        if ( mode == Qt::IgnoreAspectRatio ) {
            d->reader->setScaledSize( size );
            notScaled = false;
        } else if (d->reader->supportsOption( QImageIOHandler::Size ) ) {
            QSize actual = d->reader->size();
            actual.scale( size, mode );
            d->reader->setScaledSize( actual );
            notScaled = false;
        }
    }

    QImage thumbnail = d->reader->read();

    if( notScaled ) {
        thumbnail = thumbnail.scaled( size, mode, transformationMode );
    }

    return QPixmap::fromImage( thumbnail );
}
